import React from 'react';
import { Link } from 'react-router-dom';
import { useTranslation } from 'react-i18next';
import { Button } from '@/components/ui/button';
import LanguageSwitcher from '@/components/LanguageSwitcher';
import ThemeSwitcher from '@/components/ThemeSwitcher';
import { Package, Users, ShoppingCart, Wrench, Globe, Phone } from 'lucide-react';

const HomePage = () => {
  const { t } = useTranslation();
  
  const features = [
    {
      icon: Package,
      title: t('services.import'),
      description: t('services.import.desc'),
    },
    {
      icon: ShoppingCart,
      title: t('services.b2b'),
      description: t('services.b2b.desc'),
    },
    {
      icon: Wrench,
      title: t('services.workshop'),
      description: t('services.workshop.desc'),
    },
    {
      icon: Users,
      title: t('services.distribution'),
      description: t('services.distribution.desc'),
    },
  ];

  const brands = [
    { name: 'Bosch', logo: 'https://via.placeholder.com/120x60?text=Bosch' },
    { name: 'Mann Filter', logo: 'https://via.placeholder.com/120x60?text=Mann' },
    { name: 'NGK', logo: 'https://via.placeholder.com/120x60?text=NGK' },
    { name: 'Valeo', logo: 'https://via.placeholder.com/120x60?text=Valeo' },
  ];

  return (
    <div className="min-h-screen bg-white">
      {/* Navigation */}
      <nav className="border-b bg-white/95 backdrop-blur sticky top-0 z-50">
        <div className="container mx-auto px-6 py-4">
          <div className="flex items-center justify-between">
            <Link to="/" className="flex items-center space-x-2">
              <div className="w-10 h-10 bg-indigo-600 rounded-lg flex items-center justify-center">
                <svg className="w-6 h-6 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16m14 0h2m-2 0h-5m-9 0H3m2 0h5M9 7h1m-1 4h1m4-4h1m-1 4h1m-5 10v-5a1 1 0 011-1h2a1 1 0 011 1v5m-4 0h4" />
                </svg>
              </div>
              <span className="text-2xl font-bold text-gray-900 dark:text-white">ERP Auto Algérie</span>
            </Link>
            <div className="hidden md:flex items-center space-x-4">
              <a href="#features" className="text-gray-600 dark:text-gray-300 hover:text-indigo-600 dark:hover:text-indigo-400 transition">{t('nav.services')}</a>
              <a href="#brands" className="text-gray-600 dark:text-gray-300 hover:text-indigo-600 dark:hover:text-indigo-400 transition">{t('nav.brands')}</a>
              <a href="#contact" className="text-gray-600 dark:text-gray-300 hover:text-indigo-600 dark:hover:text-indigo-400 transition">{t('nav.contact')}</a>
              
              <div className="flex items-center gap-3 ml-4 pl-4 border-l dark:border-gray-700">
                <LanguageSwitcher />
                <ThemeSwitcher />
              </div>
              
              <Link to="/login">
                <Button className="bg-indigo-600 hover:bg-indigo-700" data-testid="login-nav-button">
                  {t('nav.login')}
                </Button>
              </Link>
            </div>
          </div>
        </div>
      </nav>

      {/* Hero Section */}
      <section className="bg-gradient-to-br from-indigo-50 via-blue-50 to-purple-50 dark:from-gray-900 dark:via-indigo-950 dark:to-purple-950 py-20" data-testid="hero-section">
        <div className="container mx-auto px-6">
          <div className="max-w-4xl mx-auto text-center">
            <h1 className="text-5xl md:text-6xl font-bold text-gray-900 dark:text-white mb-6">
              {t('hero.title')}<br />
              <span className="text-indigo-600 dark:text-indigo-400">{t('hero.subtitle')}</span>
            </h1>
            <p className="text-xl text-gray-600 dark:text-gray-300 mb-8 leading-relaxed">
              Plateforme complète pour importateurs, distributeurs de pièces auto et ateliers de réparation en Algérie.
              Gestion de stock, commandes B2B, ateliers, RH, finance et taxation algérienne.
            </p>
            <div className="flex flex-col sm:flex-row gap-4 justify-center">
              <Link to="/login">
                <Button size="lg" className="bg-indigo-600 hover:bg-indigo-700 text-lg px-8" data-testid="hero-start-button">
                  {t('hero.cta')}
                </Button>
              </Link>
              <Button size="lg" variant="outline" className="text-lg px-8 dark:border-gray-600 dark:text-gray-300">
                {t('hero.cta2')}
              </Button>
            </div>
          </div>
        </div>
      </section>

      {/* Features Section */}
      <section id="features" className="py-20 bg-white dark:bg-gray-900">
        <div className="container mx-auto px-6">
          <div className="text-center mb-16">
            <h2 className="text-4xl font-bold text-gray-900 dark:text-white mb-4">{t('services.title')}</h2>
            <p className="text-xl text-gray-600 dark:text-gray-300 max-w-2xl mx-auto">
              {t('services.subtitle')}
            </p>
          </div>
          <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-8">
            {features.map((feature, index) => {
              const Icon = feature.icon;
              return (
                <div
                  key={index}
                  className="bg-white dark:bg-gray-800 p-6 rounded-xl shadow-lg hover:shadow-xl transition-shadow border border-gray-100 dark:border-gray-700"
                  data-testid={`feature-card-${index}`}
                >
                  <div className="w-12 h-12 bg-indigo-100 dark:bg-indigo-900 rounded-lg flex items-center justify-center mb-4">
                    <Icon className="h-6 w-6 text-indigo-600 dark:text-indigo-400" />
                  </div>
                  <h3 className="text-xl font-bold text-gray-900 dark:text-white mb-2">{feature.title}</h3>
                  <p className="text-gray-600 dark:text-gray-300">{feature.description}</p>
                </div>
              );
            })}
          </div>
        </div>
      </section>

      {/* Brands Section */}
      <section id="brands" className="py-20 bg-gray-50 dark:bg-gray-800">
        <div className="container mx-auto px-6">
          <div className="text-center mb-12">
            <h2 className="text-4xl font-bold text-gray-900 mb-4">Marques Distribuées</h2>
            <p className="text-xl text-gray-600">Nous travaillons avec les meilleures marques du marché</p>
          </div>
          <div className="grid grid-cols-2 md:grid-cols-4 gap-8 items-center">
            {brands.map((brand, index) => (
              <div
                key={index}
                className="bg-white p-6 rounded-lg shadow hover:shadow-lg transition-shadow flex items-center justify-center"
              >
                <img src={brand.logo} alt={brand.name} className="max-h-12 opacity-70 hover:opacity-100 transition" />
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* CTA Section */}
      <section className="py-20 bg-indigo-600">
        <div className="container mx-auto px-6 text-center">
          <h2 className="text-4xl font-bold text-white mb-4">Prêt à commencer?</h2>
          <p className="text-xl text-indigo-100 mb-8 max-w-2xl mx-auto">
            Rejoignez les entreprises qui font confiance à notre solution ERP pour gérer leur activité
          </p>
          <Link to="/login">
            <Button size="lg" className="bg-white text-indigo-600 hover:bg-gray-100 text-lg px-8">
              Se Connecter
            </Button>
          </Link>
        </div>
      </section>

      {/* Contact Section */}
      <section id="contact" className="py-20 bg-white">
        <div className="container mx-auto px-6">
          <div className="max-w-4xl mx-auto">
            <div className="text-center mb-12">
              <h2 className="text-4xl font-bold text-gray-900 mb-4">Contactez-nous</h2>
              <p className="text-xl text-gray-600">Notre équipe est à votre disposition</p>
            </div>
            <div className="grid md:grid-cols-3 gap-8">
              <div className="text-center">
                <div className="w-12 h-12 bg-indigo-100 rounded-full flex items-center justify-center mx-auto mb-4">
                  <Phone className="h-6 w-6 text-indigo-600" />
                </div>
                <h3 className="font-bold text-gray-900 mb-2">Téléphone</h3>
                <p className="text-gray-600">+213 555 000 000</p>
              </div>
              <div className="text-center">
                <div className="w-12 h-12 bg-indigo-100 rounded-full flex items-center justify-center mx-auto mb-4">
                  <Globe className="h-6 w-6 text-indigo-600" />
                </div>
                <h3 className="font-bold text-gray-900 mb-2">Email</h3>
                <p className="text-gray-600">contact@erpauto.dz</p>
              </div>
              <div className="text-center">
                <div className="w-12 h-12 bg-indigo-100 rounded-full flex items-center justify-center mx-auto mb-4">
                  <Globe className="h-6 w-6 text-indigo-600" />
                </div>
                <h3 className="font-bold text-gray-900 mb-2">Adresse</h3>
                <p className="text-gray-600">Zone Industrielle, Alger</p>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* Footer */}
      <footer className="bg-gray-900 text-white py-12">
        <div className="container mx-auto px-6">
          <div className="text-center">
            <p className="text-gray-400">© 2025 ERP Auto Algérie. Tous droits réservés.</p>
          </div>
        </div>
      </footer>
    </div>
  );
};

export default HomePage;