import React, { useEffect, useState } from 'react';
import api from '@/services/api';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from '@/components/ui/dialog';
import { Eye, Package } from 'lucide-react';
import { format } from 'date-fns';

const ClientOrders = () => {
  const [orders, setOrders] = useState([]);
  const [selectedOrder, setSelectedOrder] = useState(null);
  const [showDetails, setShowDetails] = useState(false);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    fetchOrders();
  }, []);

  const fetchOrders = async () => {
    try {
      const response = await api.get('/orders');
      setOrders(response.data);
    } catch (error) {
      console.error('Error fetching orders:', error);
    } finally {
      setLoading(false);
    }
  };

  const getStatusBadge = (status) => {
    const config = {
      pending: { variant: 'warning', label: 'En attente' },
      confirmed: { variant: 'info', label: 'Confirmée' },
      preparing: { variant: 'info', label: 'En préparation' },
      ready: { variant: 'success', label: 'Prête' },
      in_delivery: { variant: 'info', label: 'En livraison' },
      delivered: { variant: 'success', label: 'Livrée' },
      cancelled: { variant: 'destructive', label: 'Annulée' },
    };
    const c = config[status] || config.pending;
    return <Badge variant={c.variant}>{c.label}</Badge>;
  };

  if (loading) {
    return <div className="flex items-center justify-center h-64"><div className="text-lg">Chargement...</div></div>;
  }

  return (
    <div className="space-y-6" data-testid="client-orders-page">
      <div>
        <h1 className="text-3xl font-bold text-gray-900">Mes Commandes</h1>
        <p className="text-gray-600 mt-2">{orders.length} commande(s) passée(s)</p>
      </div>

      {orders.length === 0 ? (
        <Card>
          <CardContent className="py-16 text-center">
            <Package className="h-16 w-16 mx-auto text-gray-400 mb-4" />
            <h3 className="text-xl font-semibold text-gray-900 mb-2">Aucune commande</h3>
            <p className="text-gray-600 mb-6">Vous n'avez pas encore passé de commande</p>
            <Button onClick={() => window.location.href = '/client/catalog'} className="bg-indigo-600 hover:bg-indigo-700">
              Parcourir le catalogue
            </Button>
          </CardContent>
        </Card>
      ) : (
        <div className="space-y-4">
          {orders.map((order) => (
            <Card key={order.id} data-testid={`order-card-${order.id}`}>
              <CardContent className="p-6">
                <div className="flex items-start justify-between">
                  <div className="space-y-2">
                    <div className="flex items-center space-x-4">
                      <h3 className="text-lg font-bold">{order.order_number}</h3>
                      {getStatusBadge(order.status)}
                    </div>
                    <p className="text-sm text-gray-600">
                      Date: {order.created_at ? format(new Date(order.created_at), 'dd/MM/yyyy à HH:mm') : '-'}
                    </p>
                    <p className="text-sm text-gray-600">
                      {order.items?.length || 0} article(s)
                    </p>
                  </div>

                  <div className="text-right space-y-2">
                    <p className="text-2xl font-bold text-indigo-600">
                      {new Intl.NumberFormat('fr-DZ', {
                        style: 'currency',
                        currency: 'DZD',
                      }).format(order.total)}
                    </p>
                    <Dialog open={showDetails && selectedOrder?.id === order.id} onOpenChange={(open) => {
                      setShowDetails(open);
                      if (!open) setSelectedOrder(null);
                    }}>
                      <DialogTrigger asChild>
                        <Button
                          variant="outline"
                          size="sm"
                          onClick={() => setSelectedOrder(order)}
                          data-testid={`view-order-${order.id}`}
                        >
                          <Eye className="h-4 w-4 mr-2" />
                          Détails
                        </Button>
                      </DialogTrigger>
                      <DialogContent className="max-w-2xl">
                        <DialogHeader>
                          <DialogTitle>Commande {order.order_number}</DialogTitle>
                          <DialogDescription>
                            Passée le {order.created_at ? format(new Date(order.created_at), 'dd/MM/yyyy à HH:mm') : '-'}
                          </DialogDescription>
                        </DialogHeader>
                        <div className="space-y-4">
                          <div className="flex items-center justify-between">
                            <span className="text-sm text-gray-500">Statut actuel</span>
                            {getStatusBadge(order.status)}
                          </div>

                          <div>
                            <p className="text-sm text-gray-500 mb-2">Articles commandés</p>
                            <div className="border rounded-lg overflow-hidden">
                              <table className="w-full text-sm">
                                <thead className="bg-gray-50">
                                  <tr>
                                    <th className="px-4 py-2 text-left">Produit</th>
                                    <th className="px-4 py-2 text-center">Qté</th>
                                    <th className="px-4 py-2 text-right">Prix Unit.</th>
                                    <th className="px-4 py-2 text-right">Total</th>
                                  </tr>
                                </thead>
                                <tbody>
                                  {order.items?.map((item, idx) => (
                                    <tr key={idx} className="border-t">
                                      <td className="px-4 py-2">
                                        <p className="font-medium">{item.product_name}</p>
                                        <p className="text-xs text-gray-500">{item.product_reference}</p>
                                      </td>
                                      <td className="px-4 py-2 text-center">{item.quantity}</td>
                                      <td className="px-4 py-2 text-right">
                                        {new Intl.NumberFormat('fr-DZ', {
                                          style: 'currency',
                                          currency: 'DZD',
                                        }).format(item.unit_price)}
                                      </td>
                                      <td className="px-4 py-2 text-right font-semibold">
                                        {new Intl.NumberFormat('fr-DZ', {
                                          style: 'currency',
                                          currency: 'DZD',
                                        }).format(item.subtotal)}
                                      </td>
                                    </tr>
                                  ))}
                                </tbody>
                              </table>
                            </div>
                          </div>

                          <div className="border-t pt-4 space-y-2">
                            <div className="flex justify-between">
                              <span>Sous-total</span>
                              <span className="font-medium">
                                {new Intl.NumberFormat('fr-DZ', {
                                  style: 'currency',
                                  currency: 'DZD',
                                }).format(order.subtotal)}
                              </span>
                            </div>
                            <div className="flex justify-between">
                              <span>TVA (19%)</span>
                              <span className="font-medium">
                                {new Intl.NumberFormat('fr-DZ', {
                                  style: 'currency',
                                  currency: 'DZD',
                                }).format(order.tax_amount)}
                              </span>
                            </div>
                            <div className="flex justify-between text-lg font-bold border-t pt-2">
                              <span>Total TTC</span>
                              <span className="text-indigo-600">
                                {new Intl.NumberFormat('fr-DZ', {
                                  style: 'currency',
                                  currency: 'DZD',
                                }).format(order.total)}
                              </span>
                            </div>
                          </div>

                          {order.delivery_address && (
                            <div>
                              <p className="text-sm text-gray-500 mb-1">Adresse de livraison</p>
                              <p className="font-medium">{order.delivery_address}</p>
                            </div>
                          )}
                        </div>
                      </DialogContent>
                    </Dialog>
                  </div>
                </div>

                {/* Progress Bar */}
                <div className="mt-4">
                  <div className="flex items-center justify-between text-xs text-gray-500 mb-2">
                    <span>Progression</span>
                    <span>
                      {order.status === 'pending' && '20%'}
                      {order.status === 'confirmed' && '40%'}
                      {order.status === 'preparing' && '60%'}
                      {order.status === 'ready' && '80%'}
                      {(order.status === 'in_delivery' || order.status === 'delivered') && '100%'}
                    </span>
                  </div>
                  <div className="w-full bg-gray-200 rounded-full h-2">
                    <div
                      className={`h-2 rounded-full transition-all ${
                        order.status === 'delivered' ? 'bg-green-500' : 
                        order.status === 'cancelled' ? 'bg-red-500' : 'bg-indigo-600'
                      }`}
                      style={{
                        width:
                          order.status === 'pending' ? '20%' :
                          order.status === 'confirmed' ? '40%' :
                          order.status === 'preparing' ? '60%' :
                          order.status === 'ready' ? '80%' :
                          (order.status === 'in_delivery' || order.status === 'delivered') ? '100%' : '0%'
                      }}
                    />
                  </div>
                </div>
              </CardContent>
            </Card>
          ))}
        </div>
      )}
    </div>
  );
};

export default ClientOrders;
