import React, { useState, useEffect } from 'react';
import { useTranslation } from 'react-i18next';
import api from '@/services/api';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Badge } from '@/components/ui/badge';
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Textarea } from '@/components/ui/textarea';
import { Wrench, Plus, Calendar, Clock, Car, CheckCircle, Trash2, Edit, Search } from 'lucide-react';

const WorkshopManagement = () => {
  const { t } = useTranslation();
  const [repairOrders, setRepairOrders] = useState([]);
  const [loading, setLoading] = useState(true);
  const [products, setProducts] = useState([]);
  const [clients, setClients] = useState([]);
  const [searchTerm, setSearchTerm] = useState('');
  const [statusFilter, setStatusFilter] = useState('all');

  // Dialog states
  const [showDialog, setShowDialog] = useState(false);
  const [showDetailsDialog, setShowDetailsDialog] = useState(false);
  const [selectedOrder, setSelectedOrder] = useState(null);
  const [isEditing, setIsEditing] = useState(false);

  // Form state
  const [formData, setFormData] = useState({
    client_id: '',
    client_name: '',
    vehicle_make: '',
    vehicle_model: '',
    vehicle_year: '',
    vehicle_vin: '',
    vehicle_plate: '',
    mileage: '',
    priority: 'medium',
    scheduled_date: '',
    assigned_technician: '',
    description: '',
    diagnosis: '',
    notes: '',
    services: [],
    parts: []
  });

  // Service/Part form states
  const [serviceForm, setServiceForm] = useState({
    description: '',
    labor_hours: '',
    labor_rate: '1500'
  });
  
  const [partForm, setPartForm] = useState({
    product_id: '',
    quantity: 1
  });

  const technicians = ['Karim M.', 'Ahmed B.', 'Yacine K.', 'Sofiane R.'];

  useEffect(() => {
    fetchRepairOrders();
    fetchProducts();
    fetchClients();
  }, []);

  const fetchRepairOrders = async () => {
    try {
      setLoading(true);
      let url = '/repair-orders?limit=100';
      if (statusFilter && statusFilter !== 'all') url += `&status=${statusFilter}`;
      if (searchTerm) url += `&search=${searchTerm}`;
      
      const response = await api.get(url);
      setRepairOrders(response.data);
    } catch (error) {
      console.error('Error fetching repair orders:', error);
    } finally {
      setLoading(false);
    }
  };

  const fetchProducts = async () => {
    try {
      const response = await api.get('/products?limit=500');
      setProducts(response.data);
    } catch (error) {
      console.error('Error fetching products:', error);
    }
  };

  const fetchClients = async () => {
    try {
      const response = await api.get('/clients');
      setClients(response.data);
    } catch (error) {
      console.error('Error fetching clients:', error);
    }
  };

  useEffect(() => {
    const delayDebounceFn = setTimeout(() => {
      fetchRepairOrders();
    }, 500);
    return () => clearTimeout(delayDebounceFn);
  }, [searchTerm, statusFilter]);

  const resetForm = () => {
    setFormData({
      client_id: '',
      client_name: '',
      vehicle_make: '',
      vehicle_model: '',
      vehicle_year: '',
      vehicle_vin: '',
      vehicle_plate: '',
      mileage: '',
      priority: 'medium',
      scheduled_date: '',
      assigned_technician: '',
      description: '',
      diagnosis: '',
      notes: '',
      services: [],
      parts: []
    });
    setIsEditing(false);
  };

  const addService = () => {
    if (!serviceForm.description || !serviceForm.labor_hours) {
      alert('Veuillez remplir tous les champs du service');
      return;
    }

    const labor_hours = parseFloat(serviceForm.labor_hours);
    const labor_rate = parseFloat(serviceForm.labor_rate);
    const labor_cost = labor_hours * labor_rate;

    setFormData({
      ...formData,
      services: [
        ...formData.services,
        {
          description: serviceForm.description,
          labor_hours,
          labor_rate,
          labor_cost
        }
      ]
    });

    setServiceForm({ description: '', labor_hours: '', labor_rate: '1500' });
  };

  const removeService = (index) => {
    setFormData({
      ...formData,
      services: formData.services.filter((_, i) => i !== index)
    });
  };

  const addPart = () => {
    if (!partForm.product_id || !partForm.quantity) {
      alert('Veuillez sélectionner une pièce et une quantité');
      return;
    }

    const product = products.find(p => p.id === partForm.product_id);
    if (!product) return;

    const quantity = parseInt(partForm.quantity);
    const unit_price = product.prices.public; // Use appropriate price
    const subtotal = quantity * unit_price;

    setFormData({
      ...formData,
      parts: [
        ...formData.parts,
        {
          product_id: product.id,
          product_name: product.name,
          product_reference: product.reference,
          quantity,
          unit_price,
          subtotal
        }
      ]
    });

    setPartForm({ product_id: '', quantity: 1 });
  };

  const removePart = (index) => {
    setFormData({
      ...formData,
      parts: formData.parts.filter((_, i) => i !== index)
    });
  };

  const handleSubmit = async () => {
    try {
      // Validation
      if (!formData.client_name || !formData.vehicle_make || !formData.vehicle_model || !formData.vehicle_plate || !formData.description) {
        alert('Veuillez remplir tous les champs obligatoires');
        return;
      }

      const submitData = {
        ...formData,
        vehicle_year: formData.vehicle_year ? parseInt(formData.vehicle_year) : null,
        mileage: formData.mileage ? parseInt(formData.mileage) : null,
        scheduled_date: formData.scheduled_date ? new Date(formData.scheduled_date).toISOString() : null
      };

      if (isEditing && selectedOrder) {
        await api.put(`/repair-orders/${selectedOrder.id}`, submitData);
      } else {
        await api.post('/repair-orders', submitData);
      }

      fetchRepairOrders();
      setShowDialog(false);
      resetForm();
    } catch (error) {
      console.error('Error saving repair order:', error);
      alert('Erreur lors de l\'enregistrement de l\'ordre de réparation');
    }
  };

  const handleEdit = (order) => {
    setSelectedOrder(order);
    setFormData({
      client_id: order.client_id || '',
      client_name: order.client_name,
      vehicle_make: order.vehicle_make,
      vehicle_model: order.vehicle_model,
      vehicle_year: order.vehicle_year?.toString() || '',
      vehicle_vin: order.vehicle_vin || '',
      vehicle_plate: order.vehicle_plate,
      mileage: order.mileage?.toString() || '',
      priority: order.priority,
      scheduled_date: order.scheduled_date ? new Date(order.scheduled_date).toISOString().split('T')[0] : '',
      assigned_technician: order.assigned_technician || '',
      description: order.description,
      diagnosis: order.diagnosis || '',
      notes: order.notes || '',
      services: order.services || [],
      parts: order.parts || []
    });
    setIsEditing(true);
    setShowDialog(true);
  };

  const handleDelete = async (orderId) => {
    if (!window.confirm('Êtes-vous sûr de vouloir supprimer cet ordre de réparation ?')) {
      return;
    }

    try {
      await api.delete(`/repair-orders/${orderId}`);
      fetchRepairOrders();
    } catch (error) {
      console.error('Error deleting repair order:', error);
      alert('Erreur lors de la suppression');
    }
  };

  const updateStatus = async (orderId, newStatus) => {
    try {
      await api.patch(`/repair-orders/${orderId}/status`, { status: newStatus });
      fetchRepairOrders();
      if (selectedOrder?.id === orderId) {
        setSelectedOrder({ ...selectedOrder, status: newStatus });
      }
    } catch (error) {
      console.error('Error updating status:', error);
      alert('Erreur lors de la mise à jour du statut');
    }
  };

  const openDetailsDialog = (order) => {
    setSelectedOrder(order);
    setShowDetailsDialog(true);
  };

  const getStatusBadge = (status) => {
    const config = {
      draft: { variant: 'outline', label: 'Brouillon' },
      scheduled: { variant: 'default', label: 'Planifié' },
      in_progress: { variant: 'default', label: 'En cours' },
      waiting_parts: { variant: 'secondary', label: 'Attente pièces' },
      completed: { variant: 'default', label: 'Terminé' },
      invoiced: { variant: 'default', label: 'Facturé' },
      paid: { variant: 'default', label: 'Payé' },
      cancelled: { variant: 'destructive', label: 'Annulé' }
    };
    const c = config[status] || config.draft;
    return <Badge variant={c.variant}>{c.label}</Badge>;
  };

  const getPriorityBadge = (priority) => {
    const config = {
      low: { variant: 'outline', label: 'Basse' },
      medium: { variant: 'default', label: 'Moyenne' },
      high: { variant: 'secondary', label: 'Haute' },
      urgent: { variant: 'destructive', label: 'Urgent' }
    };
    const c = config[priority] || config.medium;
    return <Badge variant={c.variant}>{c.label}</Badge>;
  };

  const stats = {
    total: repairOrders.length,
    in_progress: repairOrders.filter(o => o.status === 'in_progress').length,
    completed: repairOrders.filter(o => o.status === 'completed').length,
    revenue: repairOrders
      .filter(o => ['completed', 'invoiced', 'paid'].includes(o.status))
      .reduce((sum, o) => sum + (o.total || 0), 0)
  };

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold text-gray-900">Gestion Atelier</h1>
          <p className="text-gray-600 mt-2">Ordres de réparation et planning mécanique</p>
        </div>
        <Button
          className="bg-indigo-600 hover:bg-indigo-700"
          onClick={() => {
            resetForm();
            setShowDialog(true);
          }}
        >
          <Plus className="h-4 w-4 mr-2" />
          Nouvel OR
        </Button>
      </div>

      {/* Stats Cards */}
      <div className="grid gap-6 md:grid-cols-4">
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Total OR</CardTitle>
            <Wrench className="h-4 w-4 text-blue-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{stats.total}</div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">En Cours</CardTitle>
            <Clock className="h-4 w-4 text-orange-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-orange-600">{stats.in_progress}</div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Terminés</CardTitle>
            <CheckCircle className="h-4 w-4 text-green-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-green-600">{stats.completed}</div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">CA Atelier</CardTitle>
            <Wrench className="h-4 w-4 text-purple-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-purple-600">
              {new Intl.NumberFormat('fr-DZ', {
                style: 'currency',
                currency: 'DZD',
                minimumFractionDigits: 0
              }).format(stats.revenue)}
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Filters */}
      <Card>
        <CardContent className="pt-6">
          <div className="flex gap-4">
            <div className="flex-1">
              <Input
                placeholder="Rechercher par N° OR, client, plaque, VIN..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="w-full"
              />
            </div>
            <Select value={statusFilter} onValueChange={setStatusFilter}>
              <SelectTrigger className="w-48">
                <SelectValue placeholder="Tous les statuts" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">Tous les statuts</SelectItem>
                <SelectItem value="draft">Brouillon</SelectItem>
                <SelectItem value="scheduled">Planifié</SelectItem>
                <SelectItem value="in_progress">En cours</SelectItem>
                <SelectItem value="waiting_parts">Attente pièces</SelectItem>
                <SelectItem value="completed">Terminé</SelectItem>
                <SelectItem value="invoiced">Facturé</SelectItem>
                <SelectItem value="paid">Payé</SelectItem>
              </SelectContent>
            </Select>
          </div>
        </CardContent>
      </Card>

      {/* Repair Orders Table */}
      <Card>
        <CardHeader>
          <CardTitle>Ordres de Réparation</CardTitle>
          <CardDescription>Liste des véhicules en atelier</CardDescription>
        </CardHeader>
        <CardContent>
          {loading ? (
            <div className="text-center py-8">Chargement...</div>
          ) : repairOrders.length === 0 ? (
            <div className="text-center py-8 text-gray-500">Aucun ordre de réparation trouvé</div>
          ) : (
            <div className="overflow-x-auto">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>N° OR</TableHead>
                    <TableHead>Client</TableHead>
                    <TableHead>Véhicule</TableHead>
                    <TableHead>Plaque</TableHead>
                    <TableHead>Technicien</TableHead>
                    <TableHead>Date prévue</TableHead>
                    <TableHead>Priorité</TableHead>
                    <TableHead className="text-right">Total</TableHead>
                    <TableHead>Statut</TableHead>
                    <TableHead className="text-center">Actions</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {repairOrders.map((order) => (
                    <TableRow key={order.id}>
                      <TableCell className="font-medium">{order.repair_number}</TableCell>
                      <TableCell>{order.client_name}</TableCell>
                      <TableCell>
                        <div className="flex items-center gap-2">
                          <Car className="h-4 w-4 text-gray-400" />
                          {order.vehicle_make} {order.vehicle_model}
                          {order.vehicle_year && ` (${order.vehicle_year})`}
                        </div>
                      </TableCell>
                      <TableCell className="font-mono text-sm">{order.vehicle_plate}</TableCell>
                      <TableCell>{order.assigned_technician || '-'}</TableCell>
                      <TableCell>
                        {order.scheduled_date
                          ? new Date(order.scheduled_date).toLocaleDateString('fr-FR')
                          : '-'}
                      </TableCell>
                      <TableCell>{getPriorityBadge(order.priority)}</TableCell>
                      <TableCell className="text-right font-bold text-green-600">
                        {new Intl.NumberFormat('fr-DZ', {
                          style: 'currency',
                          currency: 'DZD'
                        }).format(order.total)}
                      </TableCell>
                      <TableCell>{getStatusBadge(order.status)}</TableCell>
                      <TableCell>
                        <div className="flex gap-2 justify-center">
                          <Button
                            variant="outline"
                            size="sm"
                            onClick={() => openDetailsDialog(order)}
                          >
                            Détails
                          </Button>
                          <Button
                            variant="outline"
                            size="sm"
                            onClick={() => handleEdit(order)}
                          >
                            <Edit className="h-4 w-4" />
                          </Button>
                          <Button
                            variant="destructive"
                            size="sm"
                            onClick={() => handleDelete(order.id)}
                          >
                            <Trash2 className="h-4 w-4" />
                          </Button>
                        </div>
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </div>
          )}
        </CardContent>
      </Card>

      {/* Create/Edit Dialog */}
      <Dialog open={showDialog} onOpenChange={setShowDialog}>
        <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>
              {isEditing ? 'Modifier l\'Ordre de Réparation' : 'Nouvel Ordre de Réparation'}
            </DialogTitle>
            <DialogDescription>
              Enregistrer les informations du véhicule et les travaux à effectuer
            </DialogDescription>
          </DialogHeader>

          <div className="space-y-6">
            {/* Client Info */}
            <div>
              <h3 className="font-semibold mb-3">Informations Client</h3>
              <div className="grid grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label>Client *</Label>
                  <Input
                    value={formData.client_name}
                    onChange={(e) => setFormData({ ...formData, client_name: e.target.value })}
                    placeholder="Nom du client"
                  />
                </div>
                <div className="space-y-2">
                  <Label>Client existant (optionnel)</Label>
                  <Select
                    value={formData.client_id}
                    onValueChange={(value) => {
                      const client = clients.find(c => c.id === value);
                      if (client) {
                        setFormData({
                          ...formData,
                          client_id: value,
                          client_name: client.company_name
                        });
                      }
                    }}
                  >
                    <SelectTrigger>
                      <SelectValue placeholder="Sélectionner un client" />
                    </SelectTrigger>
                    <SelectContent>
                      {clients.map((client) => (
                        <SelectItem key={client.id} value={client.id}>
                          {client.company_name}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>
              </div>
            </div>

            {/* Vehicle Info */}
            <div>
              <h3 className="font-semibold mb-3">Informations Véhicule</h3>
              <div className="grid grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label>Marque *</Label>
                  <Input
                    value={formData.vehicle_make}
                    onChange={(e) => setFormData({ ...formData, vehicle_make: e.target.value })}
                    placeholder="Ex: Renault"
                  />
                </div>
                <div className="space-y-2">
                  <Label>Modèle *</Label>
                  <Input
                    value={formData.vehicle_model}
                    onChange={(e) => setFormData({ ...formData, vehicle_model: e.target.value })}
                    placeholder="Ex: Clio"
                  />
                </div>
                <div className="space-y-2">
                  <Label>Année</Label>
                  <Input
                    type="number"
                    value={formData.vehicle_year}
                    onChange={(e) => setFormData({ ...formData, vehicle_year: e.target.value })}
                    placeholder="Ex: 2020"
                  />
                </div>
                <div className="space-y-2">
                  <Label>Immatriculation *</Label>
                  <Input
                    value={formData.vehicle_plate}
                    onChange={(e) => setFormData({ ...formData, vehicle_plate: e.target.value })}
                    placeholder="Ex: 16-12345-16"
                  />
                </div>
                <div className="space-y-2">
                  <Label>VIN (optionnel)</Label>
                  <Input
                    value={formData.vehicle_vin}
                    onChange={(e) => setFormData({ ...formData, vehicle_vin: e.target.value })}
                    placeholder="Numéro de châssis"
                  />
                </div>
                <div className="space-y-2">
                  <Label>Kilométrage</Label>
                  <Input
                    type="number"
                    value={formData.mileage}
                    onChange={(e) => setFormData({ ...formData, mileage: e.target.value })}
                    placeholder="Ex: 50000"
                  />
                </div>
              </div>
            </div>

            {/* Work Details */}
            <div>
              <h3 className="font-semibold mb-3">Détails du Travail</h3>
              <div className="grid grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label>Date prévue</Label>
                  <Input
                    type="date"
                    value={formData.scheduled_date}
                    onChange={(e) => setFormData({ ...formData, scheduled_date: e.target.value })}
                  />
                </div>
                <div className="space-y-2">
                  <Label>Technicien</Label>
                  <Select
                    value={formData.assigned_technician}
                    onValueChange={(value) => setFormData({ ...formData, assigned_technician: value })}
                  >
                    <SelectTrigger>
                      <SelectValue placeholder="Sélectionner" />
                    </SelectTrigger>
                    <SelectContent>
                      {technicians.map((tech) => (
                        <SelectItem key={tech} value={tech}>
                          {tech}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>
                <div className="space-y-2">
                  <Label>Priorité</Label>
                  <Select
                    value={formData.priority}
                    onValueChange={(value) => setFormData({ ...formData, priority: value })}
                  >
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="low">Basse</SelectItem>
                      <SelectItem value="medium">Moyenne</SelectItem>
                      <SelectItem value="high">Haute</SelectItem>
                      <SelectItem value="urgent">Urgent</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
                <div className="space-y-2 col-span-2">
                  <Label>Description du problème *</Label>
                  <Textarea
                    value={formData.description}
                    onChange={(e) => setFormData({ ...formData, description: e.target.value })}
                    placeholder="Décrire le problème ou la demande du client"
                    rows={3}
                  />
                </div>
                <div className="space-y-2 col-span-2">
                  <Label>Diagnostic (optionnel)</Label>
                  <Textarea
                    value={formData.diagnosis}
                    onChange={(e) => setFormData({ ...formData, diagnosis: e.target.value })}
                    placeholder="Diagnostic technique"
                    rows={2}
                  />
                </div>
              </div>
            </div>

            {/* Services/Labor */}
            <div>
              <h3 className="font-semibold mb-3">Main d'œuvre</h3>
              <div className="space-y-4">
                <div className="grid grid-cols-12 gap-2">
                  <div className="col-span-5">
                    <Input
                      value={serviceForm.description}
                      onChange={(e) => setServiceForm({ ...serviceForm, description: e.target.value })}
                      placeholder="Description du service"
                    />
                  </div>
                  <div className="col-span-2">
                    <Input
                      type="number"
                      step="0.5"
                      value={serviceForm.labor_hours}
                      onChange={(e) => setServiceForm({ ...serviceForm, labor_hours: e.target.value })}
                      placeholder="Heures"
                    />
                  </div>
                  <div className="col-span-3">
                    <Input
                      type="number"
                      value={serviceForm.labor_rate}
                      onChange={(e) => setServiceForm({ ...serviceForm, labor_rate: e.target.value })}
                      placeholder="Taux horaire"
                    />
                  </div>
                  <div className="col-span-2">
                    <Button type="button" onClick={addService} className="w-full">
                      <Plus className="h-4 w-4" />
                    </Button>
                  </div>
                </div>

                {formData.services.length > 0 && (
                  <div className="border rounded-lg p-4">
                    {formData.services.map((service, index) => (
                      <div key={index} className="flex items-center justify-between py-2 border-b last:border-0">
                        <div className="flex-1">
                          <p className="font-medium">{service.description}</p>
                          <p className="text-sm text-gray-500">
                            {service.labor_hours}h × {service.labor_rate} DZD = {service.labor_cost} DZD
                          </p>
                        </div>
                        <Button
                          type="button"
                          variant="destructive"
                          size="sm"
                          onClick={() => removeService(index)}
                        >
                          <Trash2 className="h-4 w-4" />
                        </Button>
                      </div>
                    ))}
                    <div className="mt-3 pt-3 border-t font-bold">
                      Total Main d'œuvre: {formData.services.reduce((sum, s) => sum + s.labor_cost, 0)} DZD
                    </div>
                  </div>
                )}
              </div>
            </div>

            {/* Parts */}
            <div>
              <h3 className="font-semibold mb-3">Pièces</h3>
              <div className="space-y-4">
                <div className="grid grid-cols-12 gap-2">
                  <div className="col-span-8">
                    <Select
                      value={partForm.product_id}
                      onValueChange={(value) => setPartForm({ ...partForm, product_id: value })}
                    >
                      <SelectTrigger>
                        <SelectValue placeholder="Sélectionner une pièce" />
                      </SelectTrigger>
                      <SelectContent>
                        {products.map((product) => (
                          <SelectItem key={product.id} value={product.id}>
                            {product.reference} - {product.name} ({product.prices.public} DZD)
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                  </div>
                  <div className="col-span-2">
                    <Input
                      type="number"
                      min="1"
                      value={partForm.quantity}
                      onChange={(e) => setPartForm({ ...partForm, quantity: e.target.value })}
                      placeholder="Qté"
                    />
                  </div>
                  <div className="col-span-2">
                    <Button type="button" onClick={addPart} className="w-full">
                      <Plus className="h-4 w-4" />
                    </Button>
                  </div>
                </div>

                {formData.parts.length > 0 && (
                  <div className="border rounded-lg p-4">
                    {formData.parts.map((part, index) => (
                      <div key={index} className="flex items-center justify-between py-2 border-b last:border-0">
                        <div className="flex-1">
                          <p className="font-medium">{part.product_name}</p>
                          <p className="text-sm text-gray-500">
                            Réf: {part.product_reference} | Qté: {part.quantity} × {part.unit_price} DZD = {part.subtotal} DZD
                          </p>
                        </div>
                        <Button
                          type="button"
                          variant="destructive"
                          size="sm"
                          onClick={() => removePart(index)}
                        >
                          <Trash2 className="h-4 w-4" />
                        </Button>
                      </div>
                    ))}
                    <div className="mt-3 pt-3 border-t font-bold">
                      Total Pièces: {formData.parts.reduce((sum, p) => sum + p.subtotal, 0)} DZD
                    </div>
                  </div>
                )}
              </div>
            </div>

            {/* Notes */}
            <div className="space-y-2">
              <Label>Notes (optionnel)</Label>
              <Textarea
                value={formData.notes}
                onChange={(e) => setFormData({ ...formData, notes: e.target.value })}
                placeholder="Notes additionnelles"
                rows={2}
              />
            </div>
          </div>

          <DialogFooter>
            <Button
              variant="outline"
              onClick={() => {
                setShowDialog(false);
                resetForm();
              }}
            >
              Annuler
            </Button>
            <Button onClick={handleSubmit} className="bg-indigo-600">
              {isEditing ? 'Mettre à jour' : 'Créer OR'}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Details Dialog */}
      <Dialog open={showDetailsDialog} onOpenChange={setShowDetailsDialog}>
        <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
          {selectedOrder && (
            <>
              <DialogHeader>
                <DialogTitle>Détails OR - {selectedOrder.repair_number}</DialogTitle>
                <DialogDescription>
                  {selectedOrder.client_name} - {selectedOrder.vehicle_make} {selectedOrder.vehicle_model}
                </DialogDescription>
              </DialogHeader>

              <div className="space-y-6">
                {/* Vehicle Info */}
                <Card>
                  <CardHeader>
                    <CardTitle className="text-base">Informations Véhicule</CardTitle>
                  </CardHeader>
                  <CardContent>
                    <div className="grid grid-cols-2 gap-4">
                      <div>
                        <p className="text-sm text-gray-500">Immatriculation</p>
                        <p className="font-mono font-bold">{selectedOrder.vehicle_plate}</p>
                      </div>
                      <div>
                        <p className="text-sm text-gray-500">VIN</p>
                        <p className="font-mono text-sm">{selectedOrder.vehicle_vin || '-'}</p>
                      </div>
                      <div>
                        <p className="text-sm text-gray-500">Technicien</p>
                        <p className="font-semibold">{selectedOrder.assigned_technician || '-'}</p>
                      </div>
                      <div>
                        <p className="text-sm text-gray-500">Statut</p>
                        {getStatusBadge(selectedOrder.status)}
                      </div>
                      <div>
                        <p className="text-sm text-gray-500">Priorité</p>
                        {getPriorityBadge(selectedOrder.priority)}
                      </div>
                      <div>
                        <p className="text-sm text-gray-500">Kilométrage</p>
                        <p className="font-semibold">{selectedOrder.mileage ? `${selectedOrder.mileage} km` : '-'}</p>
                      </div>
                    </div>
                  </CardContent>
                </Card>

                {/* Description/Diagnosis */}
                {(selectedOrder.description || selectedOrder.diagnosis) && (
                  <Card>
                    <CardHeader>
                      <CardTitle className="text-base">Description & Diagnostic</CardTitle>
                    </CardHeader>
                    <CardContent className="space-y-3">
                      {selectedOrder.description && (
                        <div>
                          <p className="text-sm text-gray-500">Problème</p>
                          <p>{selectedOrder.description}</p>
                        </div>
                      )}
                      {selectedOrder.diagnosis && (
                        <div>
                          <p className="text-sm text-gray-500">Diagnostic</p>
                          <p>{selectedOrder.diagnosis}</p>
                        </div>
                      )}
                    </CardContent>
                  </Card>
                )}

                {/* Services */}
                {selectedOrder.services?.length > 0 && (
                  <div>
                    <h3 className="font-semibold mb-2">Main d'œuvre</h3>
                    <div className="border rounded-lg overflow-hidden">
                      <table className="w-full text-sm">
                        <thead className="bg-gray-50">
                          <tr>
                            <th className="px-4 py-2 text-left">Service</th>
                            <th className="px-4 py-2 text-center">Heures</th>
                            <th className="px-4 py-2 text-right">Taux</th>
                            <th className="px-4 py-2 text-right">Total</th>
                          </tr>
                        </thead>
                        <tbody>
                          {selectedOrder.services.map((service, idx) => (
                            <tr key={idx} className="border-t">
                              <td className="px-4 py-2">{service.description}</td>
                              <td className="px-4 py-2 text-center">{service.labor_hours}</td>
                              <td className="px-4 py-2 text-right">{service.labor_rate} DZD</td>
                              <td className="px-4 py-2 text-right font-semibold">{service.labor_cost} DZD</td>
                            </tr>
                          ))}
                          <tr className="border-t bg-gray-50 font-bold">
                            <td className="px-4 py-2" colSpan={3}>
                              Total Main d'œuvre
                            </td>
                            <td className="px-4 py-2 text-right">{selectedOrder.labor_subtotal} DZD</td>
                          </tr>
                        </tbody>
                      </table>
                    </div>
                  </div>
                )}

                {/* Parts */}
                {selectedOrder.parts?.length > 0 && (
                  <div>
                    <h3 className="font-semibold mb-2">Pièces</h3>
                    <div className="border rounded-lg overflow-hidden">
                      <table className="w-full text-sm">
                        <thead className="bg-gray-50">
                          <tr>
                            <th className="px-4 py-2 text-left">Désignation</th>
                            <th className="px-4 py-2 text-left">Référence</th>
                            <th className="px-4 py-2 text-center">Qté</th>
                            <th className="px-4 py-2 text-right">Prix Unit.</th>
                            <th className="px-4 py-2 text-right">Total</th>
                          </tr>
                        </thead>
                        <tbody>
                          {selectedOrder.parts.map((part, idx) => (
                            <tr key={idx} className="border-t">
                              <td className="px-4 py-2">{part.product_name}</td>
                              <td className="px-4 py-2 text-gray-600">{part.product_reference}</td>
                              <td className="px-4 py-2 text-center">{part.quantity}</td>
                              <td className="px-4 py-2 text-right">{part.unit_price} DZD</td>
                              <td className="px-4 py-2 text-right font-semibold">{part.subtotal} DZD</td>
                            </tr>
                          ))}
                          <tr className="border-t bg-gray-50 font-bold">
                            <td className="px-4 py-2" colSpan={4}>
                              Total Pièces
                            </td>
                            <td className="px-4 py-2 text-right">{selectedOrder.parts_subtotal} DZD</td>
                          </tr>
                        </tbody>
                      </table>
                    </div>
                  </div>
                )}

                {/* Total */}
                <div className="space-y-2 border-t pt-4">
                  <div className="flex justify-between">
                    <span>Sous-total</span>
                    <span className="font-semibold">{selectedOrder.subtotal} DZD</span>
                  </div>
                  <div className="flex justify-between">
                    <span>TVA (19%)</span>
                    <span className="font-semibold">{selectedOrder.tax_amount} DZD</span>
                  </div>
                  <div className="flex justify-between text-2xl font-bold border-t pt-2">
                    <span>TOTAL TTC</span>
                    <span className="text-green-600">{selectedOrder.total} DZD</span>
                  </div>
                </div>

                {/* Status Update */}
                <div className="flex gap-2">
                  <Select
                    value={selectedOrder.status}
                    onValueChange={(value) => updateStatus(selectedOrder.id, value)}
                  >
                    <SelectTrigger className="flex-1">
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="draft">Brouillon</SelectItem>
                      <SelectItem value="scheduled">Planifié</SelectItem>
                      <SelectItem value="in_progress">En cours</SelectItem>
                      <SelectItem value="waiting_parts">Attente pièces</SelectItem>
                      <SelectItem value="completed">Terminé</SelectItem>
                      <SelectItem value="invoiced">Facturé</SelectItem>
                      <SelectItem value="paid">Payé</SelectItem>
                      <SelectItem value="cancelled">Annulé</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
              </div>

              <DialogFooter>
                <Button variant="outline" onClick={() => setShowDetailsDialog(false)}>
                  Fermer
                </Button>
              </DialogFooter>
            </>
          )}
        </DialogContent>
      </Dialog>
    </div>
  );
};

export default WorkshopManagement;
