import React, { useEffect, useState } from 'react';
import api from '@/services/api';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Badge } from '@/components/ui/badge';
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle, DialogTrigger } from '@/components/ui/dialog';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Warehouse, ArrowRightLeft, Plus, FileText, AlertTriangle } from 'lucide-react';

const StockManagement = () => {
  const [products, setProducts] = useState([]);
  const [loading, setLoading] = useState(true);
  const [showTransferDialog, setShowTransferDialog] = useState(false);
  const [showInventoryDialog, setShowInventoryDialog] = useState(false);
  const [selectedProduct, setSelectedProduct] = useState(null);
  const [transferData, setTransferData] = useState({
    product_id: '',
    from_site: 'principal',
    to_site: '',
    quantity: 0,
  });
  const [inventoryData, setInventoryData] = useState({
    site: '',
    notes: ''
  });

  // Mock sites data
  const sites = [
    { id: 'principal', name: 'Dépôt Principal - Alger' },
    { id: 'oran', name: 'Succursale Oran' },
    { id: 'constantine', name: 'Succursale Constantine' },
    { id: 'annaba', name: 'Succursale Annaba' },
  ];

  useEffect(() => {
    fetchProducts();
  }, []);

  const fetchProducts = async () => {
    try {
      const response = await api.get('/products');
      // Add mock multi-site stock data
      const productsWithSites = response.data.map(p => ({
        ...p,
        stock_by_site: {
          principal: p.stock_qty,
          oran: Math.floor(Math.random() * 50),
          constantine: Math.floor(Math.random() * 30),
          annaba: Math.floor(Math.random() * 20),
        }
      }));
      setProducts(productsWithSites);
    } catch (error) {
      console.error('Error fetching products:', error);
    } finally {
      setLoading(false);
    }
  };

  const handleTransfer = () => {
    // Mock transfer - in real app would call API
    alert(`Transfert de ${transferData.quantity} unités de ${transferData.from_site} vers ${transferData.to_site}`);
    setShowTransferDialog(false);
  };

  const handleStartInventory = () => {
    if (!inventoryData.site) {
      alert('Veuillez sélectionner un site');
      return;
    }
    
    // Mock inventory start - in real app would call API
    alert(`Inventaire démarré pour le site: ${sites.find(s => s.id === inventoryData.site)?.name}\n\nVous pouvez maintenant compter physiquement les stocks et enregistrer les écarts.`);
    setShowInventoryDialog(false);
    setInventoryData({ site: '', notes: '' });
  };

  const openTransferDialog = (product) => {
    setSelectedProduct(product);
    setTransferData({
      ...transferData,
      product_id: product.id,
    });
    setShowTransferDialog(true);
  };

  const lowStockProducts = products.filter(p => {
    const totalStock = Object.values(p.stock_by_site || {}).reduce((sum, qty) => sum + qty, 0);
    return totalStock <= p.min_stock;
  });

  const totalStockValue = products.reduce((sum, p) => {
    const totalQty = Object.values(p.stock_by_site || {}).reduce((s, q) => s + q, 0);
    return sum + (totalQty * p.prices.public);
  }, 0);

  if (loading) {
    return <div className="flex items-center justify-center h-64"><div className="text-lg">Chargement...</div></div>;
  }

  return (
    <div className="space-y-6" data-testid="stock-management-page">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold text-gray-900">Gestion Stock Multi-Sites</h1>
          <p className="text-gray-600 mt-2">Gérez vos stocks sur {sites.length} sites</p>
        </div>
        <div className="flex gap-2">
          <Button 
            variant="outline"
            onClick={() => setShowInventoryDialog(true)}
            data-testid="inventory-button"
          >
            <FileText className="h-4 w-4 mr-2" />
            Inventaire
          </Button>
        </div>
      </div>

      {/* Stats Cards */}
      <div className="grid gap-6 md:grid-cols-4">
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Valeur Totale Stock</CardTitle>
            <Warehouse className="h-4 w-4 text-green-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-green-600">
              {new Intl.NumberFormat('fr-DZ', {
                style: 'currency',
                currency: 'DZD',
                minimumFractionDigits: 0,
              }).format(totalStockValue)}
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Sites Actifs</CardTitle>
            <Warehouse className="h-4 w-4 text-blue-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{sites.length}</div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Alertes Stock</CardTitle>
            <AlertTriangle className="h-4 w-4 text-orange-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-orange-600">{lowStockProducts.length}</div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Produits</CardTitle>
            <Warehouse className="h-4 w-4 text-purple-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{products.length}</div>
          </CardContent>
        </Card>
      </div>

      {/* Sites Overview */}
      <div className="grid md:grid-cols-4 gap-4">
        {sites.map(site => {
          const siteStock = products.reduce((sum, p) => sum + (p.stock_by_site[site.id] || 0), 0);
          const siteValue = products.reduce((sum, p) => 
            sum + ((p.stock_by_site[site.id] || 0) * p.prices.public), 0
          );
          
          return (
            <Card key={site.id}>
              <CardHeader>
                <CardTitle className="text-base">{site.name}</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-2">
                  <div>
                    <p className="text-sm text-gray-500">Stock Total</p>
                    <p className="text-2xl font-bold">{siteStock}</p>
                  </div>
                  <div>
                    <p className="text-sm text-gray-500">Valeur</p>
                    <p className="text-lg font-semibold text-green-600">
                      {new Intl.NumberFormat('fr-DZ', {
                        style: 'currency',
                        currency: 'DZD',
                        minimumFractionDigits: 0,
                        maximumFractionDigits: 0,
                      }).format(siteValue)}
                    </p>
                  </div>
                </div>
              </CardContent>
            </Card>
          );
        })}
      </div>

      {/* Products Stock Table */}
      <Card>
        <CardHeader>
          <CardTitle>Stock par Site</CardTitle>
          <CardDescription>Vue détaillée des stocks sur tous les sites</CardDescription>
        </CardHeader>
        <CardContent>
          <div className="overflow-x-auto">
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>Produit</TableHead>
                  <TableHead className="text-center">Principal</TableHead>
                  <TableHead className="text-center">Oran</TableHead>
                  <TableHead className="text-center">Constantine</TableHead>
                  <TableHead className="text-center">Annaba</TableHead>
                  <TableHead className="text-center">Total</TableHead>
                  <TableHead className="text-center">Actions</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {products.map((product) => {
                  const totalStock = Object.values(product.stock_by_site).reduce((sum, qty) => sum + qty, 0);
                  const isLowStock = totalStock <= product.min_stock;
                  
                  return (
                    <TableRow key={product.id}>
                      <TableCell>
                        <div>
                          <p className="font-medium">{product.name}</p>
                          <p className="text-sm text-gray-500">{product.reference}</p>
                        </div>
                      </TableCell>
                      {sites.map(site => (
                        <TableCell key={site.id} className="text-center">
                          <span className={`font-semibold ${
                            product.stock_by_site[site.id] < 10 ? 'text-red-600' : 'text-green-600'
                          }`}>
                            {product.stock_by_site[site.id]}
                          </span>
                        </TableCell>
                      ))}
                      <TableCell className="text-center">
                        <span className={`font-bold ${isLowStock ? 'text-red-600' : 'text-blue-600'}`}>
                          {totalStock}
                        </span>
                      </TableCell>
                      <TableCell className="text-center">
                        <Button
                          variant="outline"
                          size="sm"
                          onClick={() => openTransferDialog(product)}
                          data-testid={`transfer-${product.id}`}
                        >
                          <ArrowRightLeft className="h-4 w-4" />
                        </Button>
                      </TableCell>
                    </TableRow>
                  );
                })}
              </TableBody>
            </Table>
          </div>
        </CardContent>
      </Card>

      {/* Transfer Dialog */}
      <Dialog open={showTransferDialog} onOpenChange={setShowTransferDialog}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Transfert Inter-Sites</DialogTitle>
            <DialogDescription>
              Transférer {selectedProduct?.name} entre sites
            </DialogDescription>
          </DialogHeader>
          <div className="grid gap-4 py-4">
            <div className="space-y-2">
              <Label>Site Source</Label>
              <Select
                value={transferData.from_site}
                onValueChange={(value) => setTransferData({...transferData, from_site: value})}
              >
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  {sites.map(site => (
                    <SelectItem key={site.id} value={site.id}>
                      {site.name} ({selectedProduct?.stock_by_site[site.id] || 0} disponibles)
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>

            <div className="space-y-2">
              <Label>Site Destination</Label>
              <Select
                value={transferData.to_site}
                onValueChange={(value) => setTransferData({...transferData, to_site: value})}
              >
                <SelectTrigger>
                  <SelectValue placeholder="Sélectionner" />
                </SelectTrigger>
                <SelectContent>
                  {sites.filter(s => s.id !== transferData.from_site).map(site => (
                    <SelectItem key={site.id} value={site.id}>{site.name}</SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>

            <div className="space-y-2">
              <Label>Quantité</Label>
              <Input
                type="number"
                value={transferData.quantity}
                onChange={(e) => setTransferData({...transferData, quantity: parseInt(e.target.value)})}
                max={selectedProduct?.stock_by_site[transferData.from_site] || 0}
              />
            </div>
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => setShowTransferDialog(false)}>Annuler</Button>
            <Button onClick={handleTransfer} className="bg-indigo-600">Transférer</Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Inventory Dialog */}
      <Dialog open={showInventoryDialog} onOpenChange={setShowInventoryDialog}>
        <DialogContent className="max-w-2xl">
          <DialogHeader>
            <DialogTitle>Inventaire Physique</DialogTitle>
            <DialogDescription>Lancer un inventaire pour un site</DialogDescription>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label>Site à inventorier</Label>
              <Select
                value={inventoryData.site}
                onValueChange={(value) => setInventoryData({...inventoryData, site: value})}
              >
                <SelectTrigger>
                  <SelectValue placeholder="Sélectionner un site" />
                </SelectTrigger>
                <SelectContent>
                  {sites.map(site => (
                    <SelectItem key={site.id} value={site.id}>{site.name}</SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <div className="space-y-2">
              <Label>Notes (optionnel)</Label>
              <Input
                value={inventoryData.notes}
                onChange={(e) => setInventoryData({...inventoryData, notes: e.target.value})}
                placeholder="Notes sur l'inventaire..."
              />
            </div>
            <div className="bg-blue-50 p-4 rounded-lg">
              <p className="text-sm text-blue-800">
                📋 L'inventaire permettra de vérifier les stocks physiques et d'ajuster les écarts.
              </p>
            </div>
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => setShowInventoryDialog(false)}>Fermer</Button>
            <Button 
              className="bg-indigo-600"
              onClick={handleStartInventory}
              data-testid="start-inventory-button"
            >
              Démarrer Inventaire
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
};

export default StockManagement;
