import React, { useEffect, useState } from 'react';
import api from '@/services/api';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { BarChart, Bar, LineChart, Line, XAxis, YAxis, CartesianGrid, Tooltip, Legend, ResponsiveContainer } from 'recharts';
import { FileText, Download, TrendingUp, Users, Package, ShoppingCart } from 'lucide-react';
import { format, subDays, startOfMonth, endOfMonth } from 'date-fns';

const ReportsPage = () => {
  const [period, setPeriod] = useState('30days');
  const [orders, setOrders] = useState([]);
  const [products, setProducts] = useState([]);
  const [clients, setClients] = useState([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    fetchData();
  }, []);

  const fetchData = async () => {
    try {
      const [ordersRes, productsRes, clientsRes] = await Promise.all([
        api.get('/orders'),
        api.get('/products'),
        api.get('/clients'),
      ]);
      setOrders(ordersRes.data);
      setProducts(productsRes.data);
      setClients(clientsRes.data);
    } catch (error) {
      console.error('Error fetching data:', error);
    } finally {
      setLoading(false);
    }
  };

  const exportToCSV = (data, filename) => {
    const csv = [
      Object.keys(data[0]).join(','),
      ...data.map(row => Object.values(row).join(','))
    ].join('\n');
    
    const blob = new Blob([csv], { type: 'text/csv' });
    const url = window.URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = filename;
    a.click();
  };

  const generateSalesReport = () => {
    const salesData = orders.map(order => ({
      'N° Commande': order.order_number,
      'Client': order.client_name,
      'Date': format(new Date(order.created_at), 'dd/MM/yyyy'),
      'Statut': order.status,
      'HT': order.subtotal,
      'TVA': order.tax_amount,
      'TTC': order.total,
    }));
    exportToCSV(salesData, 'rapport_ventes.csv');
  };

  const generateStockReport = () => {
    const stockData = products.map(product => ({
      'Référence': product.reference,
      'Nom': product.name,
      'Stock': product.stock_qty,
      'Stock Min': product.min_stock,
      'Prix Public': product.prices.public,
      'Valeur Stock': product.stock_qty * product.prices.public,
    }));
    exportToCSV(stockData, 'rapport_stock.csv');
  };

  const generateClientsReport = () => {
    const clientsData = clients.map(client => ({
      'Entreprise': client.company_name,
      'Ville': client.city || '',
      'Catégorie': client.category,
      'Limite Crédit': client.credit_limit,
      'Délai Paiement': client.payment_terms,
    }));
    exportToCSV(clientsData, 'rapport_clients.csv');
  };

  // Calculate stats
  const deliveredOrders = orders.filter(o => o.status === 'delivered');
  const totalRevenue = deliveredOrders.reduce((sum, o) => sum + o.total, 0);
  const avgOrderValue = deliveredOrders.length > 0 ? totalRevenue / deliveredOrders.length : 0;
  
  const topProducts = products
    .sort((a, b) => (b.stock_qty * b.prices.public) - (a.stock_qty * a.prices.public))
    .slice(0, 5);

  // Sales by month (mock data for demo)
  const salesByMonth = [
    { month: 'Jan', ventes: 45000, commandes: 12 },
    { month: 'Fév', ventes: 52000, commandes: 15 },
    { month: 'Mar', ventes: 48000, commandes: 13 },
    { month: 'Avr', ventes: 61000, commandes: 18 },
    { month: 'Mai', ventes: 55000, commandes: 16 },
    { month: 'Juin', ventes: 67000, commandes: 20 },
  ];

  if (loading) {
    return <div className="flex items-center justify-center h-64"><div className="text-lg">Chargement...</div></div>;
  }

  return (
    <div className="space-y-6" data-testid="reports-page">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold text-gray-900">Rapports & Analyses</h1>
          <p className="text-gray-600 mt-2">Tableaux de bord et exports</p>
        </div>
        <Select value={period} onValueChange={setPeriod}>
          <SelectTrigger className="w-48">
            <SelectValue />
          </SelectTrigger>
          <SelectContent>
            <SelectItem value="7days">7 derniers jours</SelectItem>
            <SelectItem value="30days">30 derniers jours</SelectItem>
            <SelectItem value="90days">90 derniers jours</SelectItem>
            <SelectItem value="year">Cette année</SelectItem>
          </SelectContent>
        </Select>
      </div>

      {/* Quick Stats */}
      <div className="grid gap-6 md:grid-cols-4">
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">CA Total</CardTitle>
            <TrendingUp className="h-4 w-4 text-green-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-green-600">
              {new Intl.NumberFormat('fr-DZ', {
                style: 'currency',
                currency: 'DZD',
                minimumFractionDigits: 0,
              }).format(totalRevenue)}
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Commandes</CardTitle>
            <ShoppingCart className="h-4 w-4 text-blue-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{deliveredOrders.length}</div>
            <p className="text-xs text-muted-foreground">Livrées</p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Panier Moyen</CardTitle>
            <TrendingUp className="h-4 w-4 text-indigo-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">
              {new Intl.NumberFormat('fr-DZ', {
                style: 'currency',
                currency: 'DZD',
                minimumFractionDigits: 0,
              }).format(avgOrderValue)}
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Clients Actifs</CardTitle>
            <Users className="h-4 w-4 text-purple-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{clients.length}</div>
          </CardContent>
        </Card>
      </div>

      {/* Export Buttons */}
      <div className="grid md:grid-cols-3 gap-6">
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <FileText className="h-5 w-5" />
              Rapport Ventes
            </CardTitle>
            <CardDescription>Export des ventes et commandes</CardDescription>
          </CardHeader>
          <CardContent>
            <Button
              onClick={generateSalesReport}
              className="w-full bg-green-600 hover:bg-green-700"
              data-testid="export-sales-button"
            >
              <Download className="h-4 w-4 mr-2" />
              Exporter CSV
            </Button>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Package className="h-5 w-5" />
              Rapport Stock
            </CardTitle>
            <CardDescription>État des stocks et valorisation</CardDescription>
          </CardHeader>
          <CardContent>
            <Button
              onClick={generateStockReport}
              className="w-full bg-blue-600 hover:bg-blue-700"
              data-testid="export-stock-button"
            >
              <Download className="h-4 w-4 mr-2" />
              Exporter CSV
            </Button>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Users className="h-5 w-5" />
              Rapport Clients
            </CardTitle>
            <CardDescription>Liste clients et catégories</CardDescription>
          </CardHeader>
          <CardContent>
            <Button
              onClick={generateClientsReport}
              className="w-full bg-purple-600 hover:bg-purple-700"
              data-testid="export-clients-button"
            >
              <Download className="h-4 w-4 mr-2" />
              Exporter CSV
            </Button>
          </CardContent>
        </Card>
      </div>

      {/* Charts */}
      <div className="grid md:grid-cols-2 gap-6">
        <Card>
          <CardHeader>
            <CardTitle>Évolution du CA</CardTitle>
            <CardDescription>Chiffre d'affaires mensuel (DZD)</CardDescription>
          </CardHeader>
          <CardContent>
            <ResponsiveContainer width="100%" height={300}>
              <LineChart data={salesByMonth}>
                <CartesianGrid strokeDasharray="3 3" />
                <XAxis dataKey="month" />
                <YAxis />
                <Tooltip />
                <Legend />
                <Line type="monotone" dataKey="ventes" stroke="#10b981" strokeWidth={2} name="CA (DZD)" />
              </LineChart>
            </ResponsiveContainer>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle>Commandes par Mois</CardTitle>
            <CardDescription>Nombre de commandes</CardDescription>
          </CardHeader>
          <CardContent>
            <ResponsiveContainer width="100%" height={300}>
              <BarChart data={salesByMonth}>
                <CartesianGrid strokeDasharray="3 3" />
                <XAxis dataKey="month" />
                <YAxis />
                <Tooltip />
                <Legend />
                <Bar dataKey="commandes" fill="#3b82f6" name="Commandes" />
              </BarChart>
            </ResponsiveContainer>
          </CardContent>
        </Card>
      </div>

      {/* Top Products */}
      <Card>
        <CardHeader>
          <CardTitle>Top 5 Produits (Valeur Stock)</CardTitle>
          <CardDescription>Produits avec la plus grande valeur en stock</CardDescription>
        </CardHeader>
        <CardContent>
          <div className="space-y-4">
            {topProducts.map((product, index) => (
              <div key={product.id} className="flex items-center justify-between p-4 bg-gray-50 rounded-lg">
                <div className="flex items-center gap-4">
                  <div className="flex items-center justify-center w-10 h-10 bg-indigo-100 rounded-full">
                    <span className="text-lg font-bold text-indigo-600">#{index + 1}</span>
                  </div>
                  <div>
                    <p className="font-semibold">{product.name}</p>
                    <p className="text-sm text-gray-500">Réf: {product.reference}</p>
                  </div>
                </div>
                <div className="text-right">
                  <p className="font-bold text-lg">
                    {new Intl.NumberFormat('fr-DZ', {
                      style: 'currency',
                      currency: 'DZD',
                      minimumFractionDigits: 0,
                    }).format(product.stock_qty * product.prices.public)}
                  </p>
                  <p className="text-sm text-gray-500">{product.stock_qty} unités</p>
                </div>
              </div>
            ))}
          </div>
        </CardContent>
      </Card>
    </div>
  );
};

export default ReportsPage;
