import React, { useState, useEffect } from 'react';
import api from '@/services/api';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { AlertCircle, CheckCircle, Calendar, Plus, FileText } from 'lucide-react';

const FiscalPage = () => {
  const [dashboard, setDashboard] = useState({
    upcoming_deadlines: 0,
    overdue_deadlines: 0,
    compliance_rate: 0,
    total_tva_collected: 0,
    total_tva_due: 0
  });
  const [tvaDeclarations, setTvaDeclarations] = useState([]);
  const [deadlines, setDeadlines] = useState([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    fetchDashboard();
    fetchTVADeclarations();
    fetchDeadlines();
  }, []);

  const fetchDashboard = async () => {
    try {
      const response = await api.get('/fiscal/dashboard');
      setDashboard(response.data);
    } catch (error) {
      console.error('Error:', error);
    }
  };

  const fetchTVADeclarations = async () => {
    try {
      const year = new Date().getFullYear();
      const response = await api.get(`/fiscal/tva/declarations?year=${year}`);
      setTvaDeclarations(response.data);
    } catch (error) {
      console.error('Error:', error);
    } finally {
      setLoading(false);
    }
  };

  const fetchDeadlines = async () => {
    try {
      const year = new Date().getFullYear();
      const response = await api.get(`/fiscal/deadlines?year=${year}&is_completed=false`);
      setDeadlines(response.data);
    } catch (error) {
      console.error('Error:', error);
    }
  };

  const formatCurrency = (amount) => {
    return new Intl.NumberFormat('fr-DZ', {
      style: 'currency',
      currency: 'DZD',
      minimumFractionDigits: 0
    }).format(amount);
  };

  const getStatusBadge = (status) => {
    const badges = {
      draft: { variant: 'outline', label: 'Brouillon' },
      submitted: { variant: 'default', label: 'Soumise' },
      paid: { variant: 'success', label: 'Payée' },
      late: { variant: 'destructive', label: 'En retard' }
    };
    const b = badges[status] || badges.draft;
    return <Badge variant={b.variant}>{b.label}</Badge>;
  };

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold text-gray-900">Fiscalité Algérienne</h1>
          <p className="text-gray-600 mt-2">TVA, IBS, TAP - Conformité fiscale</p>
        </div>
      </div>

      {/* Stats */}
      <div className="grid gap-6 md:grid-cols-4">
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Échéances à venir</CardTitle>
            <Calendar className="h-4 w-4 text-orange-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{dashboard.upcoming_deadlines}</div>
            <p className="text-xs text-gray-500">30 prochains jours</p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">En retard</CardTitle>
            <AlertCircle className="h-4 w-4 text-red-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-red-600">{dashboard.overdue_deadlines}</div>
            <p className="text-xs text-gray-500">Déclarations</p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Conformité</CardTitle>
            <CheckCircle className="h-4 w-4 text-green-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-green-600">{dashboard.compliance_rate}%</div>
            <p className="text-xs text-gray-500">Année en cours</p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">TVA Collectée</CardTitle>
            <FileText className="h-4 w-4 text-blue-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{formatCurrency(dashboard.total_tva_collected)}</div>
            <p className="text-xs text-gray-500">Année en cours</p>
          </CardContent>
        </Card>
      </div>

      <Tabs defaultValue="tva">
        <TabsList>
          <TabsTrigger value="tva">TVA (19%)</TabsTrigger>
          <TabsTrigger value="ibs">IBS (26%)</TabsTrigger>
          <TabsTrigger value="tap">TAP (2%)</TabsTrigger>
          <TabsTrigger value="calendar">Calendrier</TabsTrigger>
        </TabsList>

        <TabsContent value="tva" className="space-y-4">
          <Card>
            <CardHeader>
              <div className="flex items-center justify-between">
                <div>
                  <CardTitle>Déclarations TVA</CardTitle>
                  <CardDescription>Série G - Mensuelle (échéance 20 du mois)</CardDescription>
                </div>
                <Button className="bg-indigo-600">
                  <Plus className="h-4 w-4 mr-2" />
                  Nouvelle Déclaration
                </Button>
              </div>
            </CardHeader>
            <CardContent>
              {loading ? (
                <div className="text-center py-8">Chargement...</div>
              ) : tvaDeclarations.length === 0 ? (
                <div className="text-center py-8 text-gray-500">
                  Aucune déclaration TVA pour cette année
                </div>
              ) : (
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>Période</TableHead>
                      <TableHead className="text-right">Ventes HT</TableHead>
                      <TableHead className="text-right">TVA Collectée</TableHead>
                      <TableHead className="text-right">Achats HT</TableHead>
                      <TableHead className="text-right">TVA Récup.</TableHead>
                      <TableHead className="text-right">TVA Due</TableHead>
                      <TableHead>Échéance</TableHead>
                      <TableHead>Statut</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {tvaDeclarations.map((decl) => (
                      <TableRow key={decl.id}>
                        <TableCell className="font-medium">
                          {decl.period_month}/{decl.period_year}
                        </TableCell>
                        <TableCell className="text-right">{formatCurrency(decl.sales_ht)}</TableCell>
                        <TableCell className="text-right font-semibold text-green-600">
                          {formatCurrency(decl.tva_collected)}
                        </TableCell>
                        <TableCell className="text-right">{formatCurrency(decl.purchases_ht)}</TableCell>
                        <TableCell className="text-right font-semibold text-blue-600">
                          {formatCurrency(decl.tva_recoverable)}
                        </TableCell>
                        <TableCell className="text-right font-bold text-red-600">
                          {formatCurrency(decl.tva_due)}
                        </TableCell>
                        <TableCell>
                          {new Date(decl.due_date).toLocaleDateString('fr-FR')}
                        </TableCell>
                        <TableCell>{getStatusBadge(decl.status)}</TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              )}
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="ibs">
          <Card>
            <CardHeader>
              <CardTitle>Déclarations IBS</CardTitle>
              <CardDescription>Acomptes trimestriels + Déclaration annuelle</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="text-center py-8 text-gray-500">
                Fonctionnalité IBS en développement
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="tap">
          <Card>
            <CardHeader>
              <CardTitle>Déclarations TAP</CardTitle>
              <CardDescription>Taxe sur l'Activité Professionnelle - 2% CA HT</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="text-center py-8 text-gray-500">
                Fonctionnalité TAP en développement
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="calendar">
          <Card>
            <CardHeader>
              <CardTitle>Calendrier Fiscal</CardTitle>
              <CardDescription>Échéances à venir</CardDescription>
            </CardHeader>
            <CardContent>
              {deadlines.length === 0 ? (
                <div className="text-center py-8 text-gray-500">
                  Aucune échéance à venir
                </div>
              ) : (
                <div className="space-y-3">
                  {deadlines.slice(0, 10).map((deadline) => (
                    <div key={deadline.id} className="flex items-center justify-between p-3 border rounded-lg">
                      <div>
                        <p className="font-medium">{deadline.name}</p>
                        <p className="text-sm text-gray-500">
                          Échéance: {new Date(deadline.due_date).toLocaleDateString('fr-FR')}
                        </p>
                      </div>
                      <Badge variant={deadline.is_completed ? "success" : "default"}>
                        {deadline.is_completed ? "Complétée" : "En attente"}
                      </Badge>
                    </div>
                  ))}
                </div>
              )}
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  );
};

export default FiscalPage;
