import React, { useState, useEffect } from 'react';
import api from '@/services/api';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Badge } from '@/components/ui/badge';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { AlertCircle, CheckCircle, Calendar, Plus, FileText } from 'lucide-react';

const FiscalPage = () => {
  const [dashboard, setDashboard] = useState({
    upcoming_deadlines: 0,
    overdue_deadlines: 0,
    compliance_rate: 0,
    total_tva_collected: 0,
    total_tva_due: 0
  });
  const [tvaDeclarations, setTvaDeclarations] = useState([]);
  const [deadlines, setDeadlines] = useState([]);
  const [loading, setLoading] = useState(true);
  
  // Dialogs
  const [showTVADialog, setShowTVADialog] = useState(false);
  const [showGenerateCalendarDialog, setShowGenerateCalendarDialog] = useState(false);
  
  // Form
  const [tvaForm, setTvaForm] = useState({
    period_month: new Date().getMonth() + 1,
    period_year: new Date().getFullYear(),
    sales_ht: 0,
    purchases_ht: 0
  });
  
  const [calendarYear, setCalendarYear] = useState(new Date().getFullYear());

  useEffect(() => {
    fetchDashboard();
    fetchTVADeclarations();
    fetchDeadlines();
  }, []);

  const fetchDashboard = async () => {
    try {
      const response = await api.get('/fiscal/dashboard');
      setDashboard(response.data);
    } catch (error) {
      console.error('Error:', error);
    }
  };

  const fetchTVADeclarations = async () => {
    try {
      const year = new Date().getFullYear();
      const response = await api.get(`/fiscal/tva/declarations?year=${year}`);
      setTvaDeclarations(response.data);
    } catch (error) {
      console.error('Error:', error);
    } finally {
      setLoading(false);
    }
  };

  const fetchDeadlines = async () => {
    try {
      const year = new Date().getFullYear();
      const response = await api.get(`/fiscal/deadlines?year=${year}&is_completed=false`);
      setDeadlines(response.data);
    } catch (error) {
      console.error('Error:', error);
    }
  };

  const handleCreateTVA = async () => {
    try {
      await api.post('/fiscal/tva/declarations', tvaForm);
      setShowTVADialog(false);
      resetTVAForm();
      fetchTVADeclarations();
      fetchDashboard();
      alert('Déclaration TVA créée avec succès');
    } catch (error) {
      console.error('Error:', error);
      alert(error.response?.data?.detail || 'Erreur lors de la création');
    }
  };

  const handleSubmitTVA = async (declarationId) => {
    if (!window.confirm('Soumettre cette déclaration TVA ?')) return;
    
    try {
      await api.patch(`/fiscal/tva/declarations/${declarationId}/submit`);
      fetchTVADeclarations();
      fetchDashboard();
      alert('Déclaration soumise avec succès');
    } catch (error) {
      console.error('Error:', error);
      alert('Erreur lors de la soumission');
    }
  };

  const handleGenerateCalendar = async () => {
    try {
      await api.post(`/fiscal/deadlines/generate/${calendarYear}`);
      setShowGenerateCalendarDialog(false);
      fetchDeadlines();
      alert(`Calendrier fiscal ${calendarYear} généré avec succès`);
    } catch (error) {
      console.error('Error:', error);
      alert(error.response?.data?.detail || 'Erreur lors de la génération');
    }
  };

  const resetTVAForm = () => {
    setTvaForm({
      period_month: new Date().getMonth() + 1,
      period_year: new Date().getFullYear(),
      sales_ht: 0,
      purchases_ht: 0
    });
  };

  const formatCurrency = (amount) => {
    return new Intl.NumberFormat('fr-DZ', {
      style: 'currency',
      currency: 'DZD',
      minimumFractionDigits: 0
    }).format(amount);
  };

  const getStatusBadge = (status) => {
    const badges = {
      draft: { variant: 'outline', label: 'Brouillon' },
      submitted: { variant: 'default', label: 'Soumise' },
      paid: { variant: 'success', label: 'Payée' },
      late: { variant: 'destructive', label: 'En retard' }
    };
    const b = badges[status] || badges.draft;
    return <Badge variant={b.variant}>{b.label}</Badge>;
  };

  const months = [
    'Janvier', 'Février', 'Mars', 'Avril', 'Mai', 'Juin',
    'Juillet', 'Août', 'Septembre', 'Octobre', 'Novembre', 'Décembre'
  ];

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold text-gray-900">Fiscalité Algérienne</h1>
          <p className="text-gray-600 mt-2">TVA, IBS, TAP - Conformité fiscale</p>
        </div>
      </div>

      {/* Stats */}
      <div className="grid gap-6 md:grid-cols-4">
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Échéances à venir</CardTitle>
            <Calendar className="h-4 w-4 text-orange-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{dashboard.upcoming_deadlines}</div>
            <p className="text-xs text-gray-500">30 prochains jours</p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">En retard</CardTitle>
            <AlertCircle className="h-4 w-4 text-red-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-red-600">{dashboard.overdue_deadlines}</div>
            <p className="text-xs text-gray-500">Déclarations</p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Conformité</CardTitle>
            <CheckCircle className="h-4 w-4 text-green-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-green-600">{dashboard.compliance_rate}%</div>
            <p className="text-xs text-gray-500">Année en cours</p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">TVA Collectée</CardTitle>
            <FileText className="h-4 w-4 text-blue-500" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{formatCurrency(dashboard.total_tva_collected)}</div>
            <p className="text-xs text-gray-500">Année en cours</p>
          </CardContent>
        </Card>
      </div>

      <Tabs defaultValue="tva">
        <TabsList>
          <TabsTrigger value="tva">TVA (19%)</TabsTrigger>
          <TabsTrigger value="ibs">IBS (26%)</TabsTrigger>
          <TabsTrigger value="tap">TAP (2%)</TabsTrigger>
          <TabsTrigger value="calendar">Calendrier</TabsTrigger>
        </TabsList>

        <TabsContent value="tva" className="space-y-4">
          <Card>
            <CardHeader>
              <div className="flex items-center justify-between">
                <div>
                  <CardTitle>Déclarations TVA</CardTitle>
                  <CardDescription>Série G - Mensuelle (échéance 20 du mois)</CardDescription>
                </div>
                <Button 
                  className="bg-indigo-600"
                  onClick={() => setShowTVADialog(true)}
                >
                  <Plus className="h-4 w-4 mr-2" />
                  Nouvelle Déclaration
                </Button>
              </div>
            </CardHeader>
            <CardContent>
              {loading ? (
                <div className="text-center py-8">Chargement...</div>
              ) : tvaDeclarations.length === 0 ? (
                <div className="text-center py-8 text-gray-500">
                  Aucune déclaration TVA pour cette année
                </div>
              ) : (
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>Période</TableHead>
                      <TableHead className="text-right">Ventes HT</TableHead>
                      <TableHead className="text-right">TVA Collectée</TableHead>
                      <TableHead className="text-right">Achats HT</TableHead>
                      <TableHead className="text-right">TVA Récup.</TableHead>
                      <TableHead className="text-right">TVA Due</TableHead>
                      <TableHead>Échéance</TableHead>
                      <TableHead>Statut</TableHead>
                      <TableHead>Actions</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {tvaDeclarations.map((decl) => (
                      <TableRow key={decl.id}>
                        <TableCell className="font-medium">
                          {months[decl.period_month - 1]} {decl.period_year}
                        </TableCell>
                        <TableCell className="text-right">{formatCurrency(decl.sales_ht)}</TableCell>
                        <TableCell className="text-right font-semibold text-green-600">
                          {formatCurrency(decl.tva_collected)}
                        </TableCell>
                        <TableCell className="text-right">{formatCurrency(decl.purchases_ht)}</TableCell>
                        <TableCell className="text-right font-semibold text-blue-600">
                          {formatCurrency(decl.tva_recoverable)}
                        </TableCell>
                        <TableCell className="text-right font-bold text-red-600">
                          {formatCurrency(decl.tva_due)}
                        </TableCell>
                        <TableCell>
                          {new Date(decl.due_date).toLocaleDateString('fr-FR')}
                        </TableCell>
                        <TableCell>{getStatusBadge(decl.status)}</TableCell>
                        <TableCell>
                          {decl.status === 'draft' && (
                            <Button
                              size="sm"
                              onClick={() => handleSubmitTVA(decl.id)}
                            >
                              Soumettre
                            </Button>
                          )}
                        </TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              )}
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="ibs">
          <Card>
            <CardHeader>
              <CardTitle>Déclarations IBS</CardTitle>
              <CardDescription>Acomptes trimestriels + Déclaration annuelle</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="text-center py-8 text-gray-500">
                Fonctionnalité IBS en développement
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="tap">
          <Card>
            <CardHeader>
              <CardTitle>Déclarations TAP</CardTitle>
              <CardDescription>Taxe sur l'Activité Professionnelle - 2% CA HT</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="text-center py-8 text-gray-500">
                Fonctionnalité TAP en développement
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="calendar">
          <Card>
            <CardHeader>
              <div className="flex items-center justify-between">
                <div>
                  <CardTitle>Calendrier Fiscal</CardTitle>
                  <CardDescription>Échéances à venir</CardDescription>
                </div>
                <Button
                  variant="outline"
                  onClick={() => setShowGenerateCalendarDialog(true)}
                >
                  Générer Calendrier
                </Button>
              </div>
            </CardHeader>
            <CardContent>
              {deadlines.length === 0 ? (
                <div className="text-center py-8 text-gray-500">
                  Aucune échéance à venir - Générez le calendrier fiscal
                </div>
              ) : (
                <div className="space-y-3">
                  {deadlines.slice(0, 10).map((deadline) => (
                    <div key={deadline.id} className="flex items-center justify-between p-3 border rounded-lg">
                      <div>
                        <p className="font-medium">{deadline.name}</p>
                        <p className="text-sm text-gray-500">
                          Échéance: {new Date(deadline.due_date).toLocaleDateString('fr-FR')}
                        </p>
                      </div>
                      <Badge variant={deadline.is_completed ? "success" : "default"}>
                        {deadline.is_completed ? "Complétée" : "En attente"}
                      </Badge>
                    </div>
                  ))}
                </div>
              )}
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>

      {/* TVA Dialog */}
      <Dialog open={showTVADialog} onOpenChange={setShowTVADialog}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Nouvelle Déclaration TVA</DialogTitle>
            <DialogDescription>Créer une déclaration TVA mensuelle</DialogDescription>
          </DialogHeader>
          <div className="grid gap-4 py-4">
            <div className="grid grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label>Mois *</Label>
                <Select
                  value={tvaForm.period_month.toString()}
                  onValueChange={(value) => setTvaForm({...tvaForm, period_month: parseInt(value)})}
                >
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    {months.map((month, index) => (
                      <SelectItem key={index} value={(index + 1).toString()}>
                        {month}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <Label>Année *</Label>
                <Input
                  type="number"
                  value={tvaForm.period_year}
                  onChange={(e) => setTvaForm({...tvaForm, period_year: parseInt(e.target.value)})}
                />
              </div>
            </div>
            <div className="space-y-2">
              <Label>Ventes HT (DA) *</Label>
              <Input
                type="number"
                value={tvaForm.sales_ht}
                onChange={(e) => setTvaForm({...tvaForm, sales_ht: parseFloat(e.target.value) || 0})}
                placeholder="Montant total des ventes HT"
              />
              <p className="text-xs text-gray-500">
                TVA collectée (19%): {formatCurrency(tvaForm.sales_ht * 0.19)}
              </p>
            </div>
            <div className="space-y-2">
              <Label>Achats HT (DA) *</Label>
              <Input
                type="number"
                value={tvaForm.purchases_ht}
                onChange={(e) => setTvaForm({...tvaForm, purchases_ht: parseFloat(e.target.value) || 0})}
                placeholder="Montant total des achats HT"
              />
              <p className="text-xs text-gray-500">
                TVA récupérable (19%): {formatCurrency(tvaForm.purchases_ht * 0.19)}
              </p>
            </div>
            <div className="bg-blue-50 p-3 rounded">
              <p className="text-sm font-medium">TVA à payer:</p>
              <p className="text-2xl font-bold text-blue-600">
                {formatCurrency((tvaForm.sales_ht * 0.19) - (tvaForm.purchases_ht * 0.19))}
              </p>
            </div>
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => {setShowTVADialog(false); resetTVAForm();}}>
              Annuler
            </Button>
            <Button
              className="bg-indigo-600"
              onClick={handleCreateTVA}
            >
              Créer Déclaration
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Generate Calendar Dialog */}
      <Dialog open={showGenerateCalendarDialog} onOpenChange={setShowGenerateCalendarDialog}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Générer Calendrier Fiscal</DialogTitle>
            <DialogDescription>
              Génère automatiquement toutes les échéances fiscales pour une année
            </DialogDescription>
          </DialogHeader>
          <div className="py-4">
            <div className="space-y-2">
              <Label>Année</Label>
              <Input
                type="number"
                value={calendarYear}
                onChange={(e) => setCalendarYear(parseInt(e.target.value))}
                min={2020}
                max={2030}
              />
            </div>
            <div className="mt-4 bg-yellow-50 p-3 rounded">
              <p className="text-sm text-yellow-800">
                ⚠️ Ceci générera automatiquement toutes les échéances TVA, IBS et TAP pour l'année {calendarYear}
              </p>
            </div>
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => setShowGenerateCalendarDialog(false)}>
              Annuler
            </Button>
            <Button
              className="bg-indigo-600"
              onClick={handleGenerateCalendar}
            >
              Générer
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
};

export default FiscalPage;
