import React, { useState, useEffect } from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Label } from '@/components/ui/label';
import { Checkbox } from '@/components/ui/checkbox';
import { Badge } from '@/components/ui/badge';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';

const MODULES = [
  { id: 'dashboard', name: 'Tableau de Bord', icon: '📊' },
  { id: 'products', name: 'Produits', icon: '📦' },
  { id: 'stock', name: 'Stock', icon: '📋' },
  { id: 'orders', name: 'Commandes', icon: '🛒' },
  { id: 'clients', name: 'Clients', icon: '👥' },
  { id: 'workshop', name: 'Atelier', icon: '🔧' },
  { id: 'finance', name: 'Finance', icon: '💰' },
  { id: 'fiscal', name: 'Fiscalité', icon: '📑' },
  { id: 'hr', name: 'RH & Paie', icon: '👔' },
  { id: 'ged', name: 'GED', icon: '📁' },
  { id: 'reports', name: 'Rapports', icon: '📈' },
  { id: 'users', name: 'Utilisateurs', icon: '👤' },
  { id: 'imports', name: 'Imports', icon: '🚢' },
  { id: 'categories', name: 'Catégories', icon: '🏷️' },
  { id: 'brands', name: 'Marques', icon: '🏢' }
];

const ACTIONS = [
  { id: 'view', name: 'Consulter', description: 'Voir les données' },
  { id: 'create', name: 'Créer', description: 'Ajouter de nouveaux éléments' },
  { id: 'edit', name: 'Modifier', description: 'Éditer les éléments existants' },
  { id: 'delete', name: 'Supprimer', description: 'Supprimer des éléments' },
  { id: 'approve', name: 'Approuver', description: 'Valider/Approuver (congés, commandes, etc.)' }
];

const PermissionsSelector = ({ permissions, onChange, disabled = false }) => {
  const [localPermissions, setLocalPermissions] = useState(permissions || {});

  useEffect(() => {
    if (permissions) {
      setLocalPermissions(permissions);
    }
  }, [permissions]);

  const handlePermissionChange = (moduleId, action, checked) => {
    const updatedPermissions = { ...localPermissions };
    
    if (!updatedPermissions[moduleId]) {
      updatedPermissions[moduleId] = {
        view: false,
        create: false,
        edit: false,
        delete: false,
        approve: false
      };
    }
    
    updatedPermissions[moduleId][action] = checked;
    
    setLocalPermissions(updatedPermissions);
    onChange(updatedPermissions);
  };

  const handleSelectAll = (moduleId) => {
    const updatedPermissions = { ...localPermissions };
    updatedPermissions[moduleId] = {
      view: true,
      create: true,
      edit: true,
      delete: true,
      approve: true
    };
    setLocalPermissions(updatedPermissions);
    onChange(updatedPermissions);
  };

  const handleDeselectAll = (moduleId) => {
    const updatedPermissions = { ...localPermissions };
    updatedPermissions[moduleId] = {
      view: false,
      create: false,
      edit: false,
      delete: false,
      approve: false
    };
    setLocalPermissions(updatedPermissions);
    onChange(updatedPermissions);
  };

  return (
    <div className="space-y-4">
      <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
        <p className="text-sm text-blue-900">
          <strong>Important :</strong> Configurez les permissions pour chaque module. 
          Si aucune permission n'est définie, l'utilisateur aura les permissions par défaut de son rôle.
        </p>
      </div>

      <Tabs defaultValue="essential" className="w-full">
        <TabsList className="grid w-full grid-cols-3">
          <TabsTrigger value="essential">Essentiels</TabsTrigger>
          <TabsTrigger value="management">Gestion</TabsTrigger>
          <TabsTrigger value="admin">Administration</TabsTrigger>
        </TabsList>

        <TabsContent value="essential" className="space-y-3">
          {['dashboard', 'products', 'stock', 'orders', 'clients'].map(moduleId => {
            const module = MODULES.find(m => m.id === moduleId);
            const modulePerms = localPermissions[moduleId] || {};
            
            return (
              <Card key={moduleId}>
                <CardHeader className="pb-3">
                  <div className="flex items-center justify-between">
                    <div className="flex items-center gap-2">
                      <span className="text-2xl">{module.icon}</span>
                      <CardTitle className="text-base">{module.name}</CardTitle>
                    </div>
                    <div className="flex gap-2">
                      <button
                        type="button"
                        onClick={() => handleSelectAll(moduleId)}
                        disabled={disabled}
                        className="text-xs text-blue-600 hover:text-blue-700 disabled:opacity-50"
                      >
                        Tout sélectionner
                      </button>
                      <span className="text-gray-300">|</span>
                      <button
                        type="button"
                        onClick={() => handleDeselectAll(moduleId)}
                        disabled={disabled}
                        className="text-xs text-gray-600 hover:text-gray-700 disabled:opacity-50"
                      >
                        Tout déselectionner
                      </button>
                    </div>
                  </div>
                </CardHeader>
                <CardContent>
                  <div className="grid grid-cols-5 gap-3">
                    {ACTIONS.map(action => (
                      <div key={action.id} className="flex items-center space-x-2">
                        <Checkbox
                          id={`${moduleId}-${action.id}`}
                          checked={modulePerms[action.id] || false}
                          onCheckedChange={(checked) => handlePermissionChange(moduleId, action.id, checked)}
                          disabled={disabled}
                        />
                        <Label
                          htmlFor={`${moduleId}-${action.id}`}
                          className="text-sm font-normal cursor-pointer"
                          title={action.description}
                        >
                          {action.name}
                        </Label>
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>
            );
          })}
        </TabsContent>

        <TabsContent value="management" className="space-y-3">
          {['workshop', 'finance', 'fiscal', 'hr', 'ged', 'imports'].map(moduleId => {
            const module = MODULES.find(m => m.id === moduleId);
            const modulePerms = localPermissions[moduleId] || {};
            
            return (
              <Card key={moduleId}>
                <CardHeader className="pb-3">
                  <div className="flex items-center justify-between">
                    <div className="flex items-center gap-2">
                      <span className="text-2xl">{module.icon}</span>
                      <CardTitle className="text-base">{module.name}</CardTitle>
                    </div>
                    <div className="flex gap-2">
                      <button
                        type="button"
                        onClick={() => handleSelectAll(moduleId)}
                        disabled={disabled}
                        className="text-xs text-blue-600 hover:text-blue-700 disabled:opacity-50"
                      >
                        Tout sélectionner
                      </button>
                      <span className="text-gray-300">|</span>
                      <button
                        type="button"
                        onClick={() => handleDeselectAll(moduleId)}
                        disabled={disabled}
                        className="text-xs text-gray-600 hover:text-gray-700 disabled:opacity-50"
                      >
                        Tout déselectionner
                      </button>
                    </div>
                  </div>
                </CardHeader>
                <CardContent>
                  <div className="grid grid-cols-5 gap-3">
                    {ACTIONS.map(action => (
                      <div key={action.id} className="flex items-center space-x-2">
                        <Checkbox
                          id={`${moduleId}-${action.id}`}
                          checked={modulePerms[action.id] || false}
                          onCheckedChange={(checked) => handlePermissionChange(moduleId, action.id, checked)}
                          disabled={disabled}
                        />
                        <Label
                          htmlFor={`${moduleId}-${action.id}`}
                          className="text-sm font-normal cursor-pointer"
                          title={action.description}
                        >
                          {action.name}
                        </Label>
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>
            );
          })}
        </TabsContent>

        <TabsContent value="admin" className="space-y-3">
          {['reports', 'users', 'categories', 'brands'].map(moduleId => {
            const module = MODULES.find(m => m.id === moduleId);
            const modulePerms = localPermissions[moduleId] || {};
            
            return (
              <Card key={moduleId}>
                <CardHeader className="pb-3">
                  <div className="flex items-center justify-between">
                    <div className="flex items-center gap-2">
                      <span className="text-2xl">{module.icon}</span>
                      <CardTitle className="text-base">{module.name}</CardTitle>
                      {moduleId === 'users' && (
                        <Badge variant="destructive" className="text-xs">Sensible</Badge>
                      )}
                    </div>
                    <div className="flex gap-2">
                      <button
                        type="button"
                        onClick={() => handleSelectAll(moduleId)}
                        disabled={disabled}
                        className="text-xs text-blue-600 hover:text-blue-700 disabled:opacity-50"
                      >
                        Tout sélectionner
                      </button>
                      <span className="text-gray-300">|</span>
                      <button
                        type="button"
                        onClick={() => handleDeselectAll(moduleId)}
                        disabled={disabled}
                        className="text-xs text-gray-600 hover:text-gray-700 disabled:opacity-50"
                      >
                        Tout déselectionner
                      </button>
                    </div>
                  </div>
                </CardHeader>
                <CardContent>
                  <div className="grid grid-cols-5 gap-3">
                    {ACTIONS.map(action => (
                      <div key={action.id} className="flex items-center space-x-2">
                        <Checkbox
                          id={`${moduleId}-${action.id}`}
                          checked={modulePerms[action.id] || false}
                          onCheckedChange={(checked) => handlePermissionChange(moduleId, action.id, checked)}
                          disabled={disabled}
                        />
                        <Label
                          htmlFor={`${moduleId}-${action.id}`}
                          className="text-sm font-normal cursor-pointer"
                          title={action.description}
                        >
                          {action.name}
                        </Label>
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>
            );
          })}
        </TabsContent>
      </Tabs>
    </div>
  );
};

export default PermissionsSelector;
