#!/usr/bin/env python3
"""
Backend API Testing for ERP System
Tests Workshop Management (Repair Orders) and HR & Paie endpoints
"""

import requests
import json
from datetime import datetime, timezone
import sys

# Configuration
BASE_URL = "https://automotiv-erp.preview.emergentagent.com/api"
HEADERS = {"Content-Type": "application/json"}

class RepairOrderTester:
    def __init__(self):
        self.auth_token = None
        self.auth_headers = {}
        self.test_results = []
        self.created_repair_order_id = None
        
    def log_result(self, test_name, success, message, details=None):
        """Log test result"""
        status = "✅ PASS" if success else "❌ FAIL"
        print(f"{status} {test_name}: {message}")
        if details:
            print(f"   Details: {details}")
        
        self.test_results.append({
            "test": test_name,
            "success": success,
            "message": message,
            "details": details
        })
    
    def authenticate(self):
        """Get authentication token"""
        print("\n=== AUTHENTICATION ===")
        
        # Try to login with admin credentials
        login_data = {
            "email": "admin@erpauto.dz",
            "password": "admin123"
        }
        
        try:
            response = requests.post(f"{BASE_URL}/auth/login", 
                                   json=login_data, 
                                   headers=HEADERS)
            
            if response.status_code == 200:
                data = response.json()
                self.auth_token = data["access_token"]
                self.auth_headers = {
                    **HEADERS,
                    "Authorization": f"Bearer {self.auth_token}"
                }
                user_role = data["user"]["role"]
                self.log_result("Authentication", True, 
                              f"Successfully logged in as {user_role}")
                return True
            else:
                self.log_result("Authentication", False, 
                              f"Login failed: {response.status_code} - {response.text}")
                return False
                
        except Exception as e:
            self.log_result("Authentication", False, f"Login error: {str(e)}")
            return False
    
    def get_test_data(self):
        """Get existing products and clients for testing"""
        print("\n=== GETTING TEST DATA ===")
        
        # Get products
        try:
            response = requests.get(f"{BASE_URL}/products", headers=self.auth_headers)
            if response.status_code == 200:
                products = response.json()
                if products:
                    self.test_products = products[:3]  # Use first 3 products
                    self.log_result("Get Products", True, 
                                  f"Found {len(products)} products, using {len(self.test_products)} for testing")
                else:
                    self.log_result("Get Products", False, "No products found in database")
                    return False
            else:
                self.log_result("Get Products", False, f"Failed to get products: {response.status_code}")
                return False
        except Exception as e:
            self.log_result("Get Products", False, f"Error getting products: {str(e)}")
            return False
        
        # Get clients
        try:
            response = requests.get(f"{BASE_URL}/clients", headers=self.auth_headers)
            if response.status_code == 200:
                clients = response.json()
                if clients:
                    self.test_client = clients[0]  # Use first client
                    self.log_result("Get Clients", True, 
                                  f"Found {len(clients)} clients, using client: {self.test_client['company_name']}")
                else:
                    # No clients found, we'll use walk-in customer
                    self.test_client = None
                    self.log_result("Get Clients", True, "No clients found, will use walk-in customer")
            else:
                self.log_result("Get Clients", False, f"Failed to get clients: {response.status_code}")
                return False
        except Exception as e:
            self.log_result("Get Clients", False, f"Error getting clients: {str(e)}")
            return False
        
        return True
    
    def test_create_repair_order(self):
        """Test POST /api/repair-orders"""
        print("\n=== TEST CREATE REPAIR ORDER ===")
        
        # Prepare test data
        repair_order_data = {
            "client_id": self.test_client["id"] if self.test_client else None,
            "client_name": self.test_client["company_name"] if self.test_client else "Ahmed Benali",
            "vehicle_make": "Renault",
            "vehicle_model": "Clio",
            "vehicle_year": 2020,
            "vehicle_vin": "VF1RJ000123456789",
            "vehicle_plate": "16-12345-16",
            "mileage": 45000,
            "priority": "medium",
            "scheduled_date": "2024-01-15T09:00:00Z",
            "assigned_technician": "Karim Mechanic",
            "description": "Engine oil change and brake inspection",
            "diagnosis": "Routine maintenance required",
            "services": [
                {
                    "description": "Engine oil change",
                    "labor_hours": 1.0,
                    "labor_rate": 1500.0,
                    "labor_cost": 1500.0
                },
                {
                    "description": "Brake inspection",
                    "labor_hours": 0.5,
                    "labor_rate": 1500.0,
                    "labor_cost": 750.0
                }
            ],
            "parts": [],
            "notes": "Customer requested synthetic oil"
        }
        
        # Add parts from available products
        if self.test_products:
            for i, product in enumerate(self.test_products[:2]):  # Use first 2 products
                quantity = i + 1
                unit_price = product["prices"]["public"]  # Use public price
                repair_order_data["parts"].append({
                    "product_id": product["id"],
                    "product_name": product["name"],
                    "product_reference": product["reference"],
                    "quantity": quantity,
                    "unit_price": unit_price,
                    "subtotal": unit_price * quantity
                })
        
        try:
            response = requests.post(f"{BASE_URL}/repair-orders", 
                                   json=repair_order_data, 
                                   headers=self.auth_headers)
            
            if response.status_code == 200:
                repair_order = response.json()
                self.created_repair_order_id = repair_order["id"]
                
                # Verify auto-generated fields
                checks = []
                if repair_order.get("repair_number", "").startswith("REP-"):
                    checks.append("✓ Repair number auto-generated")
                else:
                    checks.append("✗ Repair number not generated correctly")
                
                if repair_order.get("labor_subtotal") == 2250.0:  # 1500 + 750
                    checks.append("✓ Labor subtotal calculated correctly")
                else:
                    checks.append(f"✗ Labor subtotal incorrect: {repair_order.get('labor_subtotal')}")
                
                expected_parts_subtotal = sum(part["subtotal"] for part in repair_order_data["parts"])
                if repair_order.get("parts_subtotal") == expected_parts_subtotal:
                    checks.append("✓ Parts subtotal calculated correctly")
                else:
                    checks.append(f"✗ Parts subtotal incorrect: {repair_order.get('parts_subtotal')}")
                
                expected_subtotal = 2250.0 + expected_parts_subtotal
                if repair_order.get("subtotal") == expected_subtotal:
                    checks.append("✓ Subtotal calculated correctly")
                else:
                    checks.append(f"✗ Subtotal incorrect: {repair_order.get('subtotal')}")
                
                expected_tax = expected_subtotal * 0.19
                if abs(repair_order.get("tax_amount", 0) - expected_tax) < 0.01:
                    checks.append("✓ Tax amount (19%) calculated correctly")
                else:
                    checks.append(f"✗ Tax amount incorrect: {repair_order.get('tax_amount')}")
                
                expected_total = expected_subtotal + expected_tax
                if abs(repair_order.get("total", 0) - expected_total) < 0.01:
                    checks.append("✓ Total calculated correctly")
                else:
                    checks.append(f"✗ Total incorrect: {repair_order.get('total')}")
                
                self.log_result("Create Repair Order", True, 
                              f"Repair order created with ID: {repair_order['id']}", 
                              "\n".join(checks))
                return True
            else:
                self.log_result("Create Repair Order", False, 
                              f"Failed to create: {response.status_code} - {response.text}")
                return False
                
        except Exception as e:
            self.log_result("Create Repair Order", False, f"Error: {str(e)}")
            return False
    
    def test_list_repair_orders(self):
        """Test GET /api/repair-orders with various filters"""
        print("\n=== TEST LIST REPAIR ORDERS ===")
        
        # Test basic listing
        try:
            response = requests.get(f"{BASE_URL}/repair-orders", headers=self.auth_headers)
            if response.status_code == 200:
                repair_orders = response.json()
                self.log_result("List All Repair Orders", True, 
                              f"Retrieved {len(repair_orders)} repair orders")
            else:
                self.log_result("List All Repair Orders", False, 
                              f"Failed: {response.status_code} - {response.text}")
                return False
        except Exception as e:
            self.log_result("List All Repair Orders", False, f"Error: {str(e)}")
            return False
        
        # Test search by repair number
        if self.created_repair_order_id:
            try:
                # Get the repair order to find its number
                response = requests.get(f"{BASE_URL}/repair-orders/{self.created_repair_order_id}", 
                                      headers=self.auth_headers)
                if response.status_code == 200:
                    repair_order = response.json()
                    repair_number = repair_order["repair_number"]
                    
                    # Search by repair number
                    response = requests.get(f"{BASE_URL}/repair-orders?search={repair_number}", 
                                          headers=self.auth_headers)
                    if response.status_code == 200:
                        search_results = response.json()
                        if any(ro["id"] == self.created_repair_order_id for ro in search_results):
                            self.log_result("Search by Repair Number", True, 
                                          f"Found repair order in search results")
                        else:
                            self.log_result("Search by Repair Number", False, 
                                          "Repair order not found in search results")
                    else:
                        self.log_result("Search by Repair Number", False, 
                                      f"Search failed: {response.status_code}")
            except Exception as e:
                self.log_result("Search by Repair Number", False, f"Error: {str(e)}")
        
        # Test status filter
        try:
            response = requests.get(f"{BASE_URL}/repair-orders?status=draft", 
                                  headers=self.auth_headers)
            if response.status_code == 200:
                draft_orders = response.json()
                self.log_result("Filter by Status", True, 
                              f"Found {len(draft_orders)} draft repair orders")
            else:
                self.log_result("Filter by Status", False, 
                              f"Status filter failed: {response.status_code}")
        except Exception as e:
            self.log_result("Filter by Status", False, f"Error: {str(e)}")
        
        return True
    
    def test_get_single_repair_order(self):
        """Test GET /api/repair-orders/{id}"""
        print("\n=== TEST GET SINGLE REPAIR ORDER ===")
        
        if not self.created_repair_order_id:
            self.log_result("Get Single Repair Order", False, "No repair order ID available")
            return False
        
        try:
            response = requests.get(f"{BASE_URL}/repair-orders/{self.created_repair_order_id}", 
                                  headers=self.auth_headers)
            
            if response.status_code == 200:
                repair_order = response.json()
                
                # Verify all required fields are present
                required_fields = ["id", "repair_number", "client_name", "vehicle_make", 
                                 "vehicle_model", "vehicle_plate", "status", "priority", 
                                 "services", "parts", "total"]
                
                missing_fields = [field for field in required_fields if field not in repair_order]
                
                if not missing_fields:
                    self.log_result("Get Single Repair Order", True, 
                                  f"Retrieved repair order with all required fields")
                else:
                    self.log_result("Get Single Repair Order", False, 
                                  f"Missing fields: {missing_fields}")
                return True
            else:
                self.log_result("Get Single Repair Order", False, 
                              f"Failed: {response.status_code} - {response.text}")
                return False
                
        except Exception as e:
            self.log_result("Get Single Repair Order", False, f"Error: {str(e)}")
            return False
    
    def test_update_repair_order(self):
        """Test PUT /api/repair-orders/{id}"""
        print("\n=== TEST UPDATE REPAIR ORDER ===")
        
        if not self.created_repair_order_id:
            self.log_result("Update Repair Order", False, "No repair order ID available")
            return False
        
        # Update data
        update_data = {
            "diagnosis": "Updated diagnosis: Oil filter also needs replacement",
            "services": [
                {
                    "description": "Engine oil change with filter",
                    "labor_hours": 1.5,
                    "labor_rate": 1500.0,
                    "labor_cost": 2250.0
                },
                {
                    "description": "Brake inspection and adjustment",
                    "labor_hours": 1.0,
                    "labor_rate": 1500.0,
                    "labor_cost": 1500.0
                }
            ],
            "notes": "Customer approved additional oil filter replacement"
        }
        
        try:
            response = requests.put(f"{BASE_URL}/repair-orders/{self.created_repair_order_id}", 
                                  json=update_data, 
                                  headers=self.auth_headers)
            
            if response.status_code == 200:
                updated_repair_order = response.json()
                
                # Verify totals were recalculated
                expected_labor_subtotal = 3750.0  # 2250 + 1500
                if updated_repair_order.get("labor_subtotal") == expected_labor_subtotal:
                    self.log_result("Update Repair Order", True, 
                                  "Repair order updated and totals recalculated correctly")
                else:
                    self.log_result("Update Repair Order", False, 
                                  f"Totals not recalculated correctly. Expected: {expected_labor_subtotal}, Got: {updated_repair_order.get('labor_subtotal')}")
                return True
            else:
                self.log_result("Update Repair Order", False, 
                              f"Failed: {response.status_code} - {response.text}")
                return False
                
        except Exception as e:
            self.log_result("Update Repair Order", False, f"Error: {str(e)}")
            return False
    
    def test_update_status(self):
        """Test PATCH /api/repair-orders/{id}/status"""
        print("\n=== TEST UPDATE STATUS ===")
        
        if not self.created_repair_order_id:
            self.log_result("Update Status", False, "No repair order ID available")
            return False
        
        # Test status transitions
        statuses_to_test = ["scheduled", "in_progress", "completed"]
        
        for status in statuses_to_test:
            try:
                status_data = {"status": status}
                response = requests.patch(f"{BASE_URL}/repair-orders/{self.created_repair_order_id}/status", 
                                        json=status_data, 
                                        headers=self.auth_headers)
                
                if response.status_code == 200:
                    # Verify status was updated
                    get_response = requests.get(f"{BASE_URL}/repair-orders/{self.created_repair_order_id}", 
                                              headers=self.auth_headers)
                    if get_response.status_code == 200:
                        repair_order = get_response.json()
                        if repair_order["status"] == status:
                            message = f"Status updated to {status}"
                            if status == "completed" and repair_order.get("completion_date"):
                                message += " (completion_date set)"
                            self.log_result(f"Update Status to {status}", True, message)
                        else:
                            self.log_result(f"Update Status to {status}", False, 
                                          f"Status not updated correctly. Expected: {status}, Got: {repair_order['status']}")
                else:
                    self.log_result(f"Update Status to {status}", False, 
                                  f"Failed: {response.status_code} - {response.text}")
                    
            except Exception as e:
                self.log_result(f"Update Status to {status}", False, f"Error: {str(e)}")
        
        return True
    
    def test_delete_repair_order(self):
        """Test DELETE /api/repair-orders/{id}"""
        print("\n=== TEST DELETE REPAIR ORDER ===")
        
        if not self.created_repair_order_id:
            self.log_result("Delete Repair Order", False, "No repair order ID available")
            return False
        
        try:
            response = requests.delete(f"{BASE_URL}/repair-orders/{self.created_repair_order_id}", 
                                     headers=self.auth_headers)
            
            if response.status_code == 200:
                # Verify deletion by trying to get the repair order
                get_response = requests.get(f"{BASE_URL}/repair-orders/{self.created_repair_order_id}", 
                                          headers=self.auth_headers)
                if get_response.status_code == 404:
                    self.log_result("Delete Repair Order", True, "Repair order deleted successfully")
                else:
                    self.log_result("Delete Repair Order", False, 
                                  "Repair order still exists after deletion")
                return True
            else:
                self.log_result("Delete Repair Order", False, 
                              f"Failed: {response.status_code} - {response.text}")
                return False
                
        except Exception as e:
            self.log_result("Delete Repair Order", False, f"Error: {str(e)}")
            return False
    
    def run_all_tests(self):
        """Run all repair order tests"""
        print("🔧 WORKSHOP MANAGEMENT API TESTING")
        print("=" * 50)
        
        # Authentication
        if not self.authenticate():
            print("\n❌ Authentication failed. Cannot proceed with tests.")
            return False
        
        # Get test data
        if not self.get_test_data():
            print("\n❌ Failed to get test data. Cannot proceed with tests.")
            return False
        
        # Run all tests
        self.test_create_repair_order()
        self.test_list_repair_orders()
        self.test_get_single_repair_order()
        self.test_update_repair_order()
        self.test_update_status()
        self.test_delete_repair_order()
        
        # Summary
        print("\n" + "=" * 50)
        print("📊 TEST SUMMARY")
        print("=" * 50)
        
        passed = sum(1 for result in self.test_results if result["success"])
        total = len(self.test_results)
        
        print(f"Total Tests: {total}")
        print(f"Passed: {passed}")
        print(f"Failed: {total - passed}")
        print(f"Success Rate: {(passed/total)*100:.1f}%")
        
        # Show failed tests
        failed_tests = [result for result in self.test_results if not result["success"]]
        if failed_tests:
            print("\n❌ FAILED TESTS:")
            for test in failed_tests:
                print(f"  - {test['test']}: {test['message']}")
        
        return passed == total

class HRTester:
    def __init__(self):
        self.auth_token = None
        self.auth_headers = {}
        self.test_results = []
        self.created_employee_id = None
        self.created_leave_request_id = None
        
    def log_result(self, test_name, success, message, details=None):
        """Log test result"""
        status = "✅ PASS" if success else "❌ FAIL"
        print(f"{status} {test_name}: {message}")
        if details:
            print(f"   Details: {details}")
        
        self.test_results.append({
            "test": test_name,
            "success": success,
            "message": message,
            "details": details
        })
    
    def authenticate(self):
        """Get authentication token"""
        print("\n=== HR AUTHENTICATION ===")
        
        # Try to login with admin credentials
        login_data = {
            "email": "admin@erpauto.dz",
            "password": "admin123"
        }
        
        try:
            response = requests.post(f"{BASE_URL}/auth/login", 
                                   json=login_data, 
                                   headers=HEADERS)
            
            if response.status_code == 200:
                data = response.json()
                self.auth_token = data["access_token"]
                self.auth_headers = {
                    **HEADERS,
                    "Authorization": f"Bearer {self.auth_token}"
                }
                user_role = data["user"]["role"]
                self.log_result("HR Authentication", True, 
                              f"Successfully logged in as {user_role}")
                return True
            else:
                self.log_result("HR Authentication", False, 
                              f"Login failed: {response.status_code} - {response.text}")
                return False
                
        except Exception as e:
            self.log_result("HR Authentication", False, f"Login error: {str(e)}")
            return False
    
    def test_hr_dashboard(self):
        """Test GET /api/hr/dashboard"""
        print("\n=== TEST HR DASHBOARD ===")
        
        try:
            response = requests.get(f"{BASE_URL}/hr/dashboard", headers=self.auth_headers)
            
            if response.status_code == 200:
                dashboard = response.json()
                
                # Verify required fields
                required_fields = ["total_employees", "departments_count", "pending_leave_requests", 
                                 "today_attendance", "upcoming_trainings"]
                
                missing_fields = [field for field in required_fields if field not in dashboard]
                
                if not missing_fields:
                    self.log_result("HR Dashboard", True, 
                                  f"Dashboard data retrieved successfully", 
                                  f"Total employees: {dashboard['total_employees']}, Departments: {dashboard['departments_count']}")
                else:
                    self.log_result("HR Dashboard", False, 
                                  f"Missing dashboard fields: {missing_fields}")
                return True
            else:
                self.log_result("HR Dashboard", False, 
                              f"Failed: {response.status_code} - {response.text}")
                return False
                
        except Exception as e:
            self.log_result("HR Dashboard", False, f"Error: {str(e)}")
            return False
    
    def test_create_employee(self):
        """Test POST /api/hr/employees"""
        print("\n=== TEST CREATE EMPLOYEE ===")
        
        # Prepare test employee data (matching EmployeeCreate model)
        employee_data = {
            "first_name": "Ahmed",
            "last_name": "Benali",
            "date_of_birth": "1985-03-15T00:00:00Z",
            "place_of_birth": "Alger",
            "gender": "M",
            "cin_number": "123456789012",
            "social_security_number": "198503151234567",
            "phone": "0555123456",
            "address": "123 Rue de la Liberté",
            "city": "Alger",
            "employee_number": "EMP001",
            "hire_date": "2024-01-15T00:00:00Z",
            "contract_type": "cdi",
            "position": "Mécanicien",
            "department": "Atelier",
            "base_salary": 45000.0
        }
        
        try:
            response = requests.post(f"{BASE_URL}/hr/employees", 
                                   json=employee_data, 
                                   headers=self.auth_headers)
            
            if response.status_code == 200:
                employee = response.json()
                self.created_employee_id = employee["id"]
                
                # Verify employee data
                checks = []
                if employee.get("employee_number") == "EMP001":
                    checks.append("✓ Employee number set correctly")
                else:
                    checks.append("✗ Employee number incorrect")
                
                if employee.get("first_name") == "Ahmed":
                    checks.append("✓ First name set correctly")
                else:
                    checks.append("✗ First name incorrect")
                
                if employee.get("base_salary") == 45000.0:
                    checks.append("✓ Base salary set correctly")
                else:
                    checks.append("✗ Base salary incorrect")
                
                if employee.get("status") == "active":
                    checks.append("✓ Default status is active")
                else:
                    checks.append("✗ Default status incorrect")
                
                self.log_result("Create Employee", True, 
                              f"Employee created with ID: {employee['id']}", 
                              "\n".join(checks))
                return True
            else:
                self.log_result("Create Employee", False, 
                              f"Failed to create: {response.status_code} - {response.text}")
                return False
                
        except Exception as e:
            self.log_result("Create Employee", False, f"Error: {str(e)}")
            return False
    
    def test_list_employees(self):
        """Test GET /api/hr/employees with filters"""
        print("\n=== TEST LIST EMPLOYEES ===")
        
        # Test basic listing
        try:
            response = requests.get(f"{BASE_URL}/hr/employees", headers=self.auth_headers)
            if response.status_code == 200:
                employees = response.json()
                self.log_result("List All Employees", True, 
                              f"Retrieved {len(employees)} employees")
            else:
                self.log_result("List All Employees", False, 
                              f"Failed: {response.status_code} - {response.text}")
                return False
        except Exception as e:
            self.log_result("List All Employees", False, f"Error: {str(e)}")
            return False
        
        # Test status filter
        try:
            response = requests.get(f"{BASE_URL}/hr/employees?status=active", 
                                  headers=self.auth_headers)
            if response.status_code == 200:
                active_employees = response.json()
                self.log_result("Filter by Status", True, 
                              f"Found {len(active_employees)} active employees")
            else:
                self.log_result("Filter by Status", False, 
                              f"Status filter failed: {response.status_code}")
        except Exception as e:
            self.log_result("Filter by Status", False, f"Error: {str(e)}")
        
        # Test department filter
        try:
            response = requests.get(f"{BASE_URL}/hr/employees?department=Atelier", 
                                  headers=self.auth_headers)
            if response.status_code == 200:
                dept_employees = response.json()
                self.log_result("Filter by Department", True, 
                              f"Found {len(dept_employees)} employees in Atelier")
            else:
                self.log_result("Filter by Department", False, 
                              f"Department filter failed: {response.status_code}")
        except Exception as e:
            self.log_result("Filter by Department", False, f"Error: {str(e)}")
        
        return True
    
    def test_get_single_employee(self):
        """Test GET /api/hr/employees/{id}"""
        print("\n=== TEST GET SINGLE EMPLOYEE ===")
        
        if not self.created_employee_id:
            self.log_result("Get Single Employee", False, "No employee ID available")
            return False
        
        try:
            response = requests.get(f"{BASE_URL}/hr/employees/{self.created_employee_id}", 
                                  headers=self.auth_headers)
            
            if response.status_code == 200:
                employee = response.json()
                
                # Verify all required fields are present
                required_fields = ["id", "first_name", "last_name", "employee_number", 
                                 "position", "department", "base_salary", "status"]
                
                missing_fields = [field for field in required_fields if field not in employee]
                
                if not missing_fields:
                    self.log_result("Get Single Employee", True, 
                                  f"Retrieved employee with all required fields")
                else:
                    self.log_result("Get Single Employee", False, 
                                  f"Missing fields: {missing_fields}")
                return True
            else:
                self.log_result("Get Single Employee", False, 
                              f"Failed: {response.status_code} - {response.text}")
                return False
                
        except Exception as e:
            self.log_result("Get Single Employee", False, f"Error: {str(e)}")
            return False
    
    def test_update_employee(self):
        """Test PUT /api/hr/employees/{id}"""
        print("\n=== TEST UPDATE EMPLOYEE ===")
        
        if not self.created_employee_id:
            self.log_result("Update Employee", False, "No employee ID available")
            return False
        
        # Update data
        update_data = {
            "position": "Mécanicien Senior",
            "base_salary": 50000.0,
            "department": "Atelier Principal"
        }
        
        try:
            response = requests.put(f"{BASE_URL}/hr/employees/{self.created_employee_id}", 
                                  json=update_data, 
                                  headers=self.auth_headers)
            
            if response.status_code == 200:
                # Verify update by getting the employee
                get_response = requests.get(f"{BASE_URL}/hr/employees/{self.created_employee_id}", 
                                          headers=self.auth_headers)
                if get_response.status_code == 200:
                    updated_employee = get_response.json()
                    if (updated_employee.get("position") == "Mécanicien Senior" and 
                        updated_employee.get("base_salary") == 50000.0):
                        self.log_result("Update Employee", True, 
                                      "Employee updated successfully")
                    else:
                        self.log_result("Update Employee", False, 
                                      "Employee data not updated correctly")
                else:
                    self.log_result("Update Employee", False, 
                                  "Could not verify update")
                return True
            else:
                self.log_result("Update Employee", False, 
                              f"Failed: {response.status_code} - {response.text}")
                return False
                
        except Exception as e:
            self.log_result("Update Employee", False, f"Error: {str(e)}")
            return False
    
    def test_create_leave_request(self):
        """Test POST /api/hr/leave-requests"""
        print("\n=== TEST CREATE LEAVE REQUEST ===")
        
        if not self.created_employee_id:
            self.log_result("Create Leave Request", False, "No employee ID available")
            return False
        
        # Prepare leave request data
        leave_data = {
            "employee_id": self.created_employee_id,
            "leave_type": "annual",
            "start_date": "2024-02-15T00:00:00Z",
            "end_date": "2024-02-20T00:00:00Z",
            "reason": "Congés annuels"
        }
        
        try:
            response = requests.post(f"{BASE_URL}/hr/leave-requests", 
                                   json=leave_data, 
                                   headers=self.auth_headers)
            
            if response.status_code == 200:
                leave_request = response.json()
                self.created_leave_request_id = leave_request["id"]
                
                # Verify auto-calculated fields
                checks = []
                if leave_request.get("days_count") == 6:  # 15-20 Feb = 6 days
                    checks.append("✓ Days count calculated correctly")
                else:
                    checks.append(f"✗ Days count incorrect: {leave_request.get('days_count')}")
                
                if leave_request.get("status") == "pending":
                    checks.append("✓ Default status is pending")
                else:
                    checks.append("✗ Default status incorrect")
                
                if leave_request.get("employee_name"):
                    checks.append("✓ Employee name populated")
                else:
                    checks.append("✗ Employee name not populated")
                
                self.log_result("Create Leave Request", True, 
                              f"Leave request created with ID: {leave_request['id']}", 
                              "\n".join(checks))
                return True
            else:
                self.log_result("Create Leave Request", False, 
                              f"Failed to create: {response.status_code} - {response.text}")
                return False
                
        except Exception as e:
            self.log_result("Create Leave Request", False, f"Error: {str(e)}")
            return False
    
    def test_list_leave_requests(self):
        """Test GET /api/hr/leave-requests with filters"""
        print("\n=== TEST LIST LEAVE REQUESTS ===")
        
        # Test basic listing
        try:
            response = requests.get(f"{BASE_URL}/hr/leave-requests", headers=self.auth_headers)
            if response.status_code == 200:
                requests_list = response.json()
                self.log_result("List All Leave Requests", True, 
                              f"Retrieved {len(requests_list)} leave requests")
            else:
                self.log_result("List All Leave Requests", False, 
                              f"Failed: {response.status_code} - {response.text}")
                return False
        except Exception as e:
            self.log_result("List All Leave Requests", False, f"Error: {str(e)}")
            return False
        
        # Test status filter
        try:
            response = requests.get(f"{BASE_URL}/hr/leave-requests?status=pending", 
                                  headers=self.auth_headers)
            if response.status_code == 200:
                pending_requests = response.json()
                self.log_result("Filter by Status", True, 
                              f"Found {len(pending_requests)} pending requests")
            else:
                self.log_result("Filter by Status", False, 
                              f"Status filter failed: {response.status_code}")
        except Exception as e:
            self.log_result("Filter by Status", False, f"Error: {str(e)}")
        
        return True
    
    def test_approve_leave_request(self):
        """Test PATCH /api/hr/leave-requests/{id}/approve"""
        print("\n=== TEST APPROVE LEAVE REQUEST ===")
        
        if not self.created_leave_request_id:
            self.log_result("Approve Leave Request", False, "No leave request ID available")
            return False
        
        try:
            response = requests.patch(f"{BASE_URL}/hr/leave-requests/{self.created_leave_request_id}/approve", 
                                    headers=self.auth_headers)
            
            if response.status_code == 200:
                # Verify approval by getting the request
                get_response = requests.get(f"{BASE_URL}/hr/leave-requests", 
                                          headers=self.auth_headers)
                if get_response.status_code == 200:
                    requests_list = get_response.json()
                    approved_request = next((req for req in requests_list 
                                           if req["id"] == self.created_leave_request_id), None)
                    
                    if approved_request and approved_request.get("status") == "approved":
                        self.log_result("Approve Leave Request", True, 
                                      "Leave request approved successfully")
                    else:
                        self.log_result("Approve Leave Request", False, 
                                      "Leave request status not updated to approved")
                else:
                    self.log_result("Approve Leave Request", False, 
                                  "Could not verify approval")
                return True
            else:
                self.log_result("Approve Leave Request", False, 
                              f"Failed: {response.status_code} - {response.text}")
                return False
                
        except Exception as e:
            self.log_result("Approve Leave Request", False, f"Error: {str(e)}")
            return False
    
    def test_generate_payslip(self):
        """Test POST /api/hr/payslips/generate"""
        print("\n=== TEST GENERATE PAYSLIP ===")
        
        if not self.created_employee_id:
            self.log_result("Generate Payslip", False, "No employee ID available")
            return False
        
        try:
            response = requests.post(f"{BASE_URL}/hr/payslips/generate?employee_id={self.created_employee_id}&period_month=1&period_year=2024", 
                                   headers=self.auth_headers)
            
            if response.status_code == 200:
                payslip = response.json()
                
                # Verify payslip calculations
                checks = []
                base_salary = 50000.0  # Updated salary from previous test
                
                if payslip.get("base_salary") == base_salary:
                    checks.append("✓ Base salary correct")
                else:
                    checks.append(f"✗ Base salary incorrect: {payslip.get('base_salary')}")
                
                expected_cnss_employee = base_salary * 0.09  # 9%
                if abs(payslip.get("cnss_employee", 0) - expected_cnss_employee) < 0.01:
                    checks.append("✓ CNSS employee (9%) calculated correctly")
                else:
                    checks.append(f"✗ CNSS employee incorrect: {payslip.get('cnss_employee')}")
                
                expected_cnss_employer = base_salary * 0.26  # 26%
                if abs(payslip.get("cnss_employer", 0) - expected_cnss_employer) < 0.01:
                    checks.append("✓ CNSS employer (26%) calculated correctly")
                else:
                    checks.append(f"✗ CNSS employer incorrect: {payslip.get('cnss_employer')}")
                
                # Check IRG calculation (simplified)
                taxable_salary = base_salary - expected_cnss_employee
                if taxable_salary <= 30000:
                    expected_irg = 0
                elif taxable_salary <= 120000:
                    expected_irg = (taxable_salary - 30000) * 0.20
                else:
                    expected_irg = (120000 - 30000) * 0.20 + (taxable_salary - 120000) * 0.35
                
                if abs(payslip.get("irg_amount", 0) - expected_irg) < 0.01:
                    checks.append("✓ IRG calculated correctly")
                else:
                    checks.append(f"✗ IRG incorrect: {payslip.get('irg_amount')} vs expected {expected_irg}")
                
                expected_net = base_salary - expected_cnss_employee - expected_irg
                if abs(payslip.get("net_salary", 0) - expected_net) < 0.01:
                    checks.append("✓ Net salary calculated correctly")
                else:
                    checks.append(f"✗ Net salary incorrect: {payslip.get('net_salary')}")
                
                self.log_result("Generate Payslip", True, 
                              f"Payslip generated successfully", 
                              "\n".join(checks))
                return True
            else:
                self.log_result("Generate Payslip", False, 
                              f"Failed to generate: {response.status_code} - {response.text}")
                return False
                
        except Exception as e:
            self.log_result("Generate Payslip", False, f"Error: {str(e)}")
            return False
    
    def test_list_payslips(self):
        """Test GET /api/hr/payslips with filters"""
        print("\n=== TEST LIST PAYSLIPS ===")
        
        # Test basic listing
        try:
            response = requests.get(f"{BASE_URL}/hr/payslips", headers=self.auth_headers)
            if response.status_code == 200:
                payslips = response.json()
                self.log_result("List All Payslips", True, 
                              f"Retrieved {len(payslips)} payslips")
            else:
                self.log_result("List All Payslips", False, 
                              f"Failed: {response.status_code} - {response.text}")
                return False
        except Exception as e:
            self.log_result("List All Payslips", False, f"Error: {str(e)}")
            return False
        
        # Test employee filter
        if self.created_employee_id:
            try:
                response = requests.get(f"{BASE_URL}/hr/payslips?employee_id={self.created_employee_id}", 
                                      headers=self.auth_headers)
                if response.status_code == 200:
                    employee_payslips = response.json()
                    self.log_result("Filter by Employee", True, 
                                  f"Found {len(employee_payslips)} payslips for employee")
                else:
                    self.log_result("Filter by Employee", False, 
                                  f"Employee filter failed: {response.status_code}")
            except Exception as e:
                self.log_result("Filter by Employee", False, f"Error: {str(e)}")
        
        return True
    
    def test_attendance_checkin_checkout(self):
        """Test POST /api/hr/attendance/check-in and check-out"""
        print("\n=== TEST ATTENDANCE CHECK-IN/OUT ===")
        
        if not self.created_employee_id:
            self.log_result("Attendance Check-in", False, "No employee ID available")
            return False
        
        # Test check-in
        try:
            response = requests.post(f"{BASE_URL}/hr/attendance/check-in?employee_id={self.created_employee_id}", 
                                   headers=self.auth_headers)
            
            if response.status_code == 200:
                checkin_result = response.json()
                self.log_result("Attendance Check-in", True, 
                              f"Check-in recorded at {checkin_result.get('time')}")
            else:
                self.log_result("Attendance Check-in", False, 
                              f"Failed: {response.status_code} - {response.text}")
                return False
        except Exception as e:
            self.log_result("Attendance Check-in", False, f"Error: {str(e)}")
            return False
        
        # Test check-out
        try:
            response = requests.post(f"{BASE_URL}/hr/attendance/check-out?employee_id={self.created_employee_id}", 
                                   headers=self.auth_headers)
            
            if response.status_code == 200:
                checkout_result = response.json()
                checks = []
                if checkout_result.get("worked_hours"):
                    checks.append(f"✓ Worked hours calculated: {checkout_result['worked_hours']}")
                if checkout_result.get("overtime_hours") is not None:
                    checks.append(f"✓ Overtime hours calculated: {checkout_result['overtime_hours']}")
                
                self.log_result("Attendance Check-out", True, 
                              f"Check-out recorded at {checkout_result.get('time')}", 
                              "\n".join(checks))
            else:
                self.log_result("Attendance Check-out", False, 
                              f"Failed: {response.status_code} - {response.text}")
                return False
        except Exception as e:
            self.log_result("Attendance Check-out", False, f"Error: {str(e)}")
            return False
        
        return True
    
    def test_terminate_employee(self):
        """Test PATCH /api/hr/employees/{id}/terminate"""
        print("\n=== TEST TERMINATE EMPLOYEE ===")
        
        if not self.created_employee_id:
            self.log_result("Terminate Employee", False, "No employee ID available")
            return False
        
        termination_data = {
            "termination_date": "2024-01-31T00:00:00Z",
            "termination_reason": "End of contract"
        }
        
        try:
            response = requests.patch(f"{BASE_URL}/hr/employees/{self.created_employee_id}/terminate", 
                                    json=termination_data, 
                                    headers=self.auth_headers)
            
            if response.status_code == 200:
                # Verify termination by getting the employee
                get_response = requests.get(f"{BASE_URL}/hr/employees/{self.created_employee_id}", 
                                          headers=self.auth_headers)
                if get_response.status_code == 200:
                    terminated_employee = get_response.json()
                    if terminated_employee.get("status") == "terminated":
                        self.log_result("Terminate Employee", True, 
                                      "Employee terminated successfully")
                    else:
                        self.log_result("Terminate Employee", False, 
                                      "Employee status not updated to terminated")
                else:
                    self.log_result("Terminate Employee", False, 
                                  "Could not verify termination")
                return True
            else:
                self.log_result("Terminate Employee", False, 
                              f"Failed: {response.status_code} - {response.text}")
                return False
                
        except Exception as e:
            self.log_result("Terminate Employee", False, f"Error: {str(e)}")
            return False
    
    def run_all_tests(self):
        """Run all HR tests"""
        print("👥 HR & PAIE API TESTING")
        print("=" * 50)
        
        # Authentication
        if not self.authenticate():
            print("\n❌ Authentication failed. Cannot proceed with tests.")
            return False
        
        # Run all tests
        self.test_hr_dashboard()
        self.test_create_employee()
        self.test_list_employees()
        self.test_get_single_employee()
        self.test_update_employee()
        self.test_create_leave_request()
        self.test_list_leave_requests()
        self.test_approve_leave_request()
        self.test_generate_payslip()
        self.test_list_payslips()
        self.test_attendance_checkin_checkout()
        self.test_terminate_employee()
        
        # Summary
        print("\n" + "=" * 50)
        print("📊 HR TEST SUMMARY")
        print("=" * 50)
        
        passed = sum(1 for result in self.test_results if result["success"])
        total = len(self.test_results)
        
        print(f"Total Tests: {total}")
        print(f"Passed: {passed}")
        print(f"Failed: {total - passed}")
        print(f"Success Rate: {(passed/total)*100:.1f}%")
        
        # Show failed tests
        failed_tests = [result for result in self.test_results if not result["success"]]
        if failed_tests:
            print("\n❌ FAILED TESTS:")
            for test in failed_tests:
                print(f"  - {test['test']}: {test['message']}")
        
        return passed == total

class PermissionsTester:
    def __init__(self):
        self.auth_token = None
        self.auth_headers = {}
        self.test_results = []
        self.created_user_id = None
        
    def log_result(self, test_name, success, message, details=None):
        """Log test result"""
        status = "✅ PASS" if success else "❌ FAIL"
        print(f"{status} {test_name}: {message}")
        if details:
            print(f"   Details: {details}")
        
        self.test_results.append({
            "test": test_name,
            "success": success,
            "message": message,
            "details": details
        })
    
    def authenticate(self):
        """Get authentication token"""
        print("\n=== PERMISSIONS AUTHENTICATION ===")
        
        # Try to login with admin credentials
        login_data = {
            "email": "admin@erpauto.dz",
            "password": "admin123"
        }
        
        try:
            response = requests.post(f"{BASE_URL}/auth/login", 
                                   json=login_data, 
                                   headers=HEADERS)
            
            if response.status_code == 200:
                data = response.json()
                self.auth_token = data["access_token"]
                self.auth_headers = {
                    **HEADERS,
                    "Authorization": f"Bearer {self.auth_token}"
                }
                user_role = data["user"]["role"]
                self.log_result("Permissions Authentication", True, 
                              f"Successfully logged in as {user_role}")
                return True
            else:
                self.log_result("Permissions Authentication", False, 
                              f"Login failed: {response.status_code} - {response.text}")
                return False
                
        except Exception as e:
            self.log_result("Permissions Authentication", False, f"Login error: {str(e)}")
            return False
    
    def test_get_permission_templates(self):
        """Test GET /api/users/templates/permissions"""
        print("\n=== TEST GET PERMISSION TEMPLATES ===")
        
        try:
            response = requests.get(f"{BASE_URL}/users/templates/permissions", 
                                  headers=self.auth_headers)
            
            if response.status_code == 200:
                templates = response.json()
                
                # Verify expected templates exist
                expected_templates = ["hr_manager", "hr_agent", "accountant", "workshop_manager", "stock_manager", "sales_manager"]
                checks = []
                
                for template in expected_templates:
                    if template in templates:
                        template_data = templates[template]
                        if "name" in template_data and "description" in template_data and "permissions" in template_data:
                            checks.append(f"✓ Template '{template}' found with complete structure")
                        else:
                            checks.append(f"✗ Template '{template}' missing required fields")
                    else:
                        checks.append(f"✗ Template '{template}' not found")
                
                # Verify hr_manager template structure
                if "hr_manager" in templates:
                    hr_template = templates["hr_manager"]["permissions"]
                    if hr_template.get("hr", {}).get("approve") == True:
                        checks.append("✓ HR Manager template has approve permission")
                    else:
                        checks.append("✗ HR Manager template missing approve permission")
                
                self.log_result("Get Permission Templates", True, 
                              f"Retrieved {len(templates)} permission templates", 
                              "\n".join(checks))
                return True
            else:
                self.log_result("Get Permission Templates", False, 
                              f"Failed: {response.status_code} - {response.text}")
                return False
                
        except Exception as e:
            self.log_result("Get Permission Templates", False, f"Error: {str(e)}")
            return False
    
    def test_list_users(self):
        """Test GET /api/users"""
        print("\n=== TEST LIST USERS ===")
        
        try:
            response = requests.get(f"{BASE_URL}/users/", headers=self.auth_headers)
            
            if response.status_code == 200:
                users = response.json()
                
                # Verify users have permissions field
                checks = []
                for user in users:
                    if "permissions" in user:
                        checks.append(f"✓ User {user.get('email', 'unknown')} has permissions field")
                    else:
                        checks.append(f"✗ User {user.get('email', 'unknown')} missing permissions field")
                
                self.log_result("List Users", True, 
                              f"Retrieved {len(users)} users", 
                              "\n".join(checks[:5]))  # Show first 5 checks
                return True
            else:
                self.log_result("List Users", False, 
                              f"Failed: {response.status_code} - {response.text}")
                return False
                
        except Exception as e:
            self.log_result("List Users", False, f"Error: {str(e)}")
            return False
    
    def test_create_user_with_permissions(self):
        """Test POST /api/auth/register with custom permissions"""
        print("\n=== TEST CREATE USER WITH PERMISSIONS ===")
        
        # Test data from the review request
        user_data = {
            "email": "test_hr@erp.dz",
            "password": "test123",
            "full_name": "Responsable RH Test",
            "phone": "0555123456",
            "role": "manager",
            "language": "fr",
            "permissions": {
                "dashboard": {"view": True, "create": False, "edit": False, "delete": False, "approve": False},
                "hr": {"view": True, "create": True, "edit": True, "delete": True, "approve": True},
                "reports": {"view": True, "create": False, "edit": False, "delete": False, "approve": False}
            }
        }
        
        try:
            response = requests.post(f"{BASE_URL}/auth/register", 
                                   json=user_data, 
                                   headers=self.auth_headers)
            
            if response.status_code == 200:
                user = response.json()
                self.created_user_id = user["id"]
                
                # Verify user creation with permissions
                checks = []
                if user.get("email") == "test_hr@erp.dz":
                    checks.append("✓ Email set correctly")
                else:
                    checks.append("✗ Email incorrect")
                
                if user.get("role") == "manager":
                    checks.append("✓ Role set correctly")
                else:
                    checks.append("✗ Role incorrect")
                
                if user.get("permissions"):
                    checks.append("✓ Custom permissions included")
                    # Verify specific permissions
                    hr_perms = user["permissions"].get("hr", {})
                    if hr_perms.get("approve") == True:
                        checks.append("✓ HR approve permission set correctly")
                    else:
                        checks.append("✗ HR approve permission not set")
                else:
                    checks.append("✗ Custom permissions missing")
                
                self.log_result("Create User with Permissions", True, 
                              f"User created with ID: {user['id']}", 
                              "\n".join(checks))
                return True
            else:
                self.log_result("Create User with Permissions", False, 
                              f"Failed to create: {response.status_code} - {response.text}")
                return False
                
        except Exception as e:
            self.log_result("Create User with Permissions", False, f"Error: {str(e)}")
            return False
    
    def test_get_single_user(self):
        """Test GET /api/users/{user_id}"""
        print("\n=== TEST GET SINGLE USER ===")
        
        if not self.created_user_id:
            self.log_result("Get Single User", False, "No user ID available")
            return False
        
        try:
            response = requests.get(f"{BASE_URL}/users/{self.created_user_id}", 
                                  headers=self.auth_headers)
            
            if response.status_code == 200:
                user = response.json()
                
                # Verify custom permissions are present
                checks = []
                if user.get("permissions"):
                    checks.append("✓ Custom permissions retrieved")
                    
                    # Verify specific permission structure
                    hr_perms = user["permissions"].get("hr", {})
                    dashboard_perms = user["permissions"].get("dashboard", {})
                    
                    if hr_perms.get("approve") == True and hr_perms.get("delete") == True:
                        checks.append("✓ HR permissions correct (approve=True, delete=True)")
                    else:
                        checks.append("✗ HR permissions incorrect")
                    
                    if dashboard_perms.get("view") == True and dashboard_perms.get("create") == False:
                        checks.append("✓ Dashboard permissions correct (view=True, create=False)")
                    else:
                        checks.append("✗ Dashboard permissions incorrect")
                else:
                    checks.append("✗ Custom permissions missing")
                
                self.log_result("Get Single User", True, 
                              f"Retrieved user with custom permissions", 
                              "\n".join(checks))
                return True
            else:
                self.log_result("Get Single User", False, 
                              f"Failed: {response.status_code} - {response.text}")
                return False
                
        except Exception as e:
            self.log_result("Get Single User", False, f"Error: {str(e)}")
            return False
    
    def test_update_user_permissions(self):
        """Test PATCH /api/users/{user_id}/permissions"""
        print("\n=== TEST UPDATE USER PERMISSIONS ===")
        
        if not self.created_user_id:
            self.log_result("Update User Permissions", False, "No user ID available")
            return False
        
        # Updated permissions
        new_permissions = {
            "dashboard": {"view": True, "create": False, "edit": False, "delete": False, "approve": False},
            "hr": {"view": True, "create": True, "edit": True, "delete": False, "approve": False},  # Removed delete and approve
            "reports": {"view": True, "create": True, "edit": False, "delete": False, "approve": False},  # Added create
            "finance": {"view": True, "create": False, "edit": False, "delete": False, "approve": False}  # Added finance
        }
        
        try:
            response = requests.patch(f"{BASE_URL}/users/{self.created_user_id}/permissions", 
                                    json=new_permissions, 
                                    headers=self.auth_headers)
            
            if response.status_code == 200:
                # Verify update by getting the user
                get_response = requests.get(f"{BASE_URL}/users/{self.created_user_id}", 
                                          headers=self.auth_headers)
                if get_response.status_code == 200:
                    updated_user = get_response.json()
                    
                    checks = []
                    hr_perms = updated_user["permissions"].get("hr", {})
                    reports_perms = updated_user["permissions"].get("reports", {})
                    finance_perms = updated_user["permissions"].get("finance", {})
                    
                    if hr_perms.get("delete") == False and hr_perms.get("approve") == False:
                        checks.append("✓ HR permissions updated (delete=False, approve=False)")
                    else:
                        checks.append("✗ HR permissions not updated correctly")
                    
                    if reports_perms.get("create") == True:
                        checks.append("✓ Reports create permission added")
                    else:
                        checks.append("✗ Reports create permission not added")
                    
                    if finance_perms.get("view") == True:
                        checks.append("✓ Finance module permissions added")
                    else:
                        checks.append("✗ Finance module permissions not added")
                    
                    self.log_result("Update User Permissions", True, 
                                  "Permissions updated successfully", 
                                  "\n".join(checks))
                else:
                    self.log_result("Update User Permissions", False, 
                                  "Could not verify update")
                return True
            else:
                self.log_result("Update User Permissions", False, 
                              f"Failed: {response.status_code} - {response.text}")
                return False
                
        except Exception as e:
            self.log_result("Update User Permissions", False, f"Error: {str(e)}")
            return False
    
    def test_update_user_status(self):
        """Test PATCH /api/users/{user_id}/status"""
        print("\n=== TEST UPDATE USER STATUS ===")
        
        if not self.created_user_id:
            self.log_result("Update User Status", False, "No user ID available")
            return False
        
        # Test deactivation
        try:
            deactivate_data = {"is_active": False}
            response = requests.patch(f"{BASE_URL}/users/{self.created_user_id}/status", 
                                    json=deactivate_data, 
                                    headers=self.auth_headers)
            
            if response.status_code == 200:
                # Verify deactivation
                get_response = requests.get(f"{BASE_URL}/users/{self.created_user_id}", 
                                          headers=self.auth_headers)
                if get_response.status_code == 200:
                    user = get_response.json()
                    if user.get("is_active") == False:
                        self.log_result("Deactivate User", True, "User deactivated successfully")
                    else:
                        self.log_result("Deactivate User", False, "User not deactivated")
                else:
                    self.log_result("Deactivate User", False, "Could not verify deactivation")
            else:
                self.log_result("Deactivate User", False, 
                              f"Failed: {response.status_code} - {response.text}")
                return False
        except Exception as e:
            self.log_result("Deactivate User", False, f"Error: {str(e)}")
            return False
        
        # Test reactivation
        try:
            activate_data = {"is_active": True}
            response = requests.patch(f"{BASE_URL}/users/{self.created_user_id}/status", 
                                    json=activate_data, 
                                    headers=self.auth_headers)
            
            if response.status_code == 200:
                # Verify reactivation
                get_response = requests.get(f"{BASE_URL}/users/{self.created_user_id}", 
                                          headers=self.auth_headers)
                if get_response.status_code == 200:
                    user = get_response.json()
                    if user.get("is_active") == True:
                        self.log_result("Reactivate User", True, "User reactivated successfully")
                    else:
                        self.log_result("Reactivate User", False, "User not reactivated")
                else:
                    self.log_result("Reactivate User", False, "Could not verify reactivation")
            else:
                self.log_result("Reactivate User", False, 
                              f"Failed: {response.status_code} - {response.text}")
                return False
        except Exception as e:
            self.log_result("Reactivate User", False, f"Error: {str(e)}")
            return False
        
        return True
    
    def test_update_user_info(self):
        """Test PUT /api/users/{user_id}"""
        print("\n=== TEST UPDATE USER INFO ===")
        
        if not self.created_user_id:
            self.log_result("Update User Info", False, "No user ID available")
            return False
        
        # Update user info (not permissions)
        update_data = {
            "full_name": "Responsable RH Test Modifié",
            "phone": "0666789012"
        }
        
        try:
            response = requests.put(f"{BASE_URL}/users/{self.created_user_id}", 
                                  json=update_data, 
                                  headers=self.auth_headers)
            
            if response.status_code == 200:
                # Verify update
                get_response = requests.get(f"{BASE_URL}/users/{self.created_user_id}", 
                                          headers=self.auth_headers)
                if get_response.status_code == 200:
                    user = get_response.json()
                    
                    checks = []
                    if user.get("full_name") == "Responsable RH Test Modifié":
                        checks.append("✓ Full name updated correctly")
                    else:
                        checks.append("✗ Full name not updated")
                    
                    if user.get("phone") == "0666789012":
                        checks.append("✓ Phone updated correctly")
                    else:
                        checks.append("✗ Phone not updated")
                    
                    # Verify permissions were not affected
                    if user.get("permissions"):
                        checks.append("✓ Permissions preserved during update")
                    else:
                        checks.append("✗ Permissions lost during update")
                    
                    self.log_result("Update User Info", True, 
                                  "User info updated successfully", 
                                  "\n".join(checks))
                else:
                    self.log_result("Update User Info", False, "Could not verify update")
                return True
            else:
                self.log_result("Update User Info", False, 
                              f"Failed: {response.status_code} - {response.text}")
                return False
                
        except Exception as e:
            self.log_result("Update User Info", False, f"Error: {str(e)}")
            return False
    
    def test_delete_user(self):
        """Test DELETE /api/users/{user_id}"""
        print("\n=== TEST DELETE USER ===")
        
        if not self.created_user_id:
            self.log_result("Delete User", False, "No user ID available")
            return False
        
        try:
            response = requests.delete(f"{BASE_URL}/users/{self.created_user_id}", 
                                     headers=self.auth_headers)
            
            if response.status_code == 200:
                # Verify deletion by trying to get the user
                get_response = requests.get(f"{BASE_URL}/users/{self.created_user_id}", 
                                          headers=self.auth_headers)
                if get_response.status_code == 404:
                    self.log_result("Delete User", True, "User deleted successfully")
                else:
                    self.log_result("Delete User", False, 
                                  "User still exists after deletion")
                return True
            else:
                self.log_result("Delete User", False, 
                              f"Failed: {response.status_code} - {response.text}")
                return False
                
        except Exception as e:
            self.log_result("Delete User", False, f"Error: {str(e)}")
            return False
    
    def test_permission_validation(self):
        """Test that only SUPERADMIN can create/modify/delete users"""
        print("\n=== TEST PERMISSION VALIDATION ===")
        
        # This test would require creating a non-superadmin user and testing restrictions
        # For now, we'll just verify that our current user (admin@erpauto.dz) has the right permissions
        
        try:
            # Test that we can access permission templates (superadmin only)
            response = requests.get(f"{BASE_URL}/users/templates/permissions", 
                                  headers=self.auth_headers)
            
            if response.status_code == 200:
                self.log_result("Permission Validation", True, 
                              "SUPERADMIN can access permission templates")
                return True
            elif response.status_code == 403:
                self.log_result("Permission Validation", True, 
                              "Non-SUPERADMIN correctly blocked from permission templates")
                return True
            else:
                self.log_result("Permission Validation", False, 
                              f"Unexpected response: {response.status_code}")
                return False
                
        except Exception as e:
            self.log_result("Permission Validation", False, f"Error: {str(e)}")
            return False
    
    def run_all_tests(self):
        """Run all permissions tests"""
        print("🔐 PERMISSIONS & USER MANAGEMENT API TESTING")
        print("=" * 60)
        
        # Authentication
        if not self.authenticate():
            print("\n❌ Authentication failed. Cannot proceed with tests.")
            return False
        
        # Run all tests
        self.test_get_permission_templates()
        self.test_list_users()
        self.test_create_user_with_permissions()
        self.test_get_single_user()
        self.test_update_user_permissions()
        self.test_update_user_status()
        self.test_update_user_info()
        self.test_delete_user()
        self.test_permission_validation()
        
        # Summary
        print("\n" + "=" * 60)
        print("📊 PERMISSIONS TEST SUMMARY")
        print("=" * 60)
        
        passed = sum(1 for result in self.test_results if result["success"])
        total = len(self.test_results)
        
        print(f"Total Tests: {total}")
        print(f"Passed: {passed}")
        print(f"Failed: {total - passed}")
        print(f"Success Rate: {(passed/total)*100:.1f}%")
        
        # Show failed tests
        failed_tests = [result for result in self.test_results if not result["success"]]
        if failed_tests:
            print("\n❌ FAILED TESTS:")
            for test in failed_tests:
                print(f"  - {test['test']}: {test['message']}")
        
        return passed == total

class AITester:
    def __init__(self):
        self.auth_token = None
        self.auth_headers = {}
        self.test_results = []
        
    def log_result(self, test_name, success, message, details=None):
        """Log test result"""
        status = "✅ PASS" if success else "❌ FAIL"
        print(f"{status} {test_name}: {message}")
        if details:
            print(f"   Details: {details}")
        
        self.test_results.append({
            "test": test_name,
            "success": success,
            "message": message,
            "details": details
        })
    
    def authenticate(self):
        """Get authentication token"""
        print("\n=== AI MODULE AUTHENTICATION ===")
        
        # Try to login with admin credentials from review request
        login_data = {
            "email": "admin@erp.dz",
            "password": "admin123"
        }
        
        try:
            response = requests.post(f"{BASE_URL}/auth/login", 
                                   json=login_data, 
                                   headers=HEADERS)
            
            if response.status_code == 200:
                data = response.json()
                self.auth_token = data["access_token"]
                self.auth_headers = {
                    **HEADERS,
                    "Authorization": f"Bearer {self.auth_token}"
                }
                user_role = data["user"]["role"]
                self.log_result("AI Authentication", True, 
                              f"Successfully logged in as {user_role}")
                return True
            else:
                self.log_result("AI Authentication", False, 
                              f"Login failed: {response.status_code} - {response.text}")
                return False
                
        except Exception as e:
            self.log_result("AI Authentication", False, f"Login error: {str(e)}")
            return False
    
    def test_general_insights(self):
        """Test GET /api/ai/insights/general"""
        print("\n=== TEST GENERAL INSIGHTS ===")
        
        try:
            response = requests.get(f"{BASE_URL}/ai/insights/general", 
                                  headers=self.auth_headers)
            
            if response.status_code == 200:
                insights = response.json()
                
                # Verify response structure
                checks = []
                if "insights" in insights:
                    checks.append("✓ Insights field present")
                    
                    insights_data = insights["insights"]
                    if isinstance(insights_data, dict):
                        if "health_score" in insights_data or "raw_response" in insights_data:
                            checks.append("✓ AI analysis generated")
                        else:
                            checks.append("✗ Missing expected AI analysis fields")
                    else:
                        checks.append("✗ Insights data not in expected format")
                else:
                    checks.append("✗ Missing insights field")
                
                if "timestamp" in insights:
                    checks.append("✓ Timestamp present")
                else:
                    checks.append("✗ Missing timestamp")
                
                self.log_result("General Insights", True, 
                              "AI insights generated successfully", 
                              "\n".join(checks))
                return True
            else:
                self.log_result("General Insights", False, 
                              f"Failed: {response.status_code} - {response.text}")
                return False
                
        except Exception as e:
            self.log_result("General Insights", False, f"Error: {str(e)}")
            return False
    
    def test_workshop_detect_anomalies(self):
        """Test POST /api/ai/workshop/detect-anomalies"""
        print("\n=== TEST WORKSHOP ANOMALY DETECTION ===")
        
        try:
            response = requests.post(f"{BASE_URL}/ai/workshop/detect-anomalies", 
                                   headers=self.auth_headers)
            
            if response.status_code == 200:
                result = response.json()
                
                checks = []
                if "anomalies" in result or "message" in result:
                    checks.append("✓ Response structure valid")
                    
                    if "anomalies" in result:
                        anomalies = result["anomalies"]
                        if isinstance(anomalies, list):
                            checks.append(f"✓ Anomalies list returned ({len(anomalies)} items)")
                        else:
                            checks.append("✗ Anomalies not in list format")
                    
                    if "message" in result:
                        checks.append(f"✓ Message: {result['message']}")
                else:
                    checks.append("✗ Invalid response structure")
                
                self.log_result("Workshop Anomaly Detection", True, 
                              "Anomaly detection completed", 
                              "\n".join(checks))
                return True
            else:
                self.log_result("Workshop Anomaly Detection", False, 
                              f"Failed: {response.status_code} - {response.text}")
                return False
                
        except Exception as e:
            self.log_result("Workshop Anomaly Detection", False, f"Error: {str(e)}")
            return False
    
    def test_workshop_analyze_repair(self):
        """Test POST /api/ai/workshop/analyze-repair"""
        print("\n=== TEST WORKSHOP REPAIR ANALYSIS ===")
        
        # Test data from review request
        repair_data = {
            "vehicle_make": "Toyota",
            "vehicle_model": "Corolla",
            "vehicle_year": "2020",
            "mileage": 85000,
            "description": "Bruit moteur, fumée anormale au démarrage",
            "services": ["Diagnostic moteur", "Changement huile"]
        }
        
        try:
            response = requests.post(f"{BASE_URL}/ai/workshop/analyze-repair", 
                                   json=repair_data,
                                   headers=self.auth_headers)
            
            if response.status_code == 200:
                result = response.json()
                
                checks = []
                if "analysis" in result:
                    checks.append("✓ Analysis field present")
                    
                    analysis = result["analysis"]
                    if isinstance(analysis, dict):
                        # Check for expected fields or raw response
                        expected_fields = ["estimated_cost", "recommended_parts", "estimated_time", "potential_issues"]
                        found_fields = [field for field in expected_fields if field in analysis]
                        
                        if found_fields:
                            checks.append(f"✓ AI analysis structured ({len(found_fields)}/4 expected fields)")
                        elif "raw_response" in analysis:
                            checks.append("✓ AI response received (unstructured)")
                        else:
                            checks.append("✗ No valid analysis data")
                    else:
                        checks.append("✗ Analysis not in expected format")
                else:
                    checks.append("✗ Missing analysis field")
                
                if "timestamp" in result:
                    checks.append("✓ Timestamp present")
                
                self.log_result("Workshop Repair Analysis", True, 
                              "Repair analysis completed", 
                              "\n".join(checks))
                return True
            else:
                self.log_result("Workshop Repair Analysis", False, 
                              f"Failed: {response.status_code} - {response.text}")
                return False
                
        except Exception as e:
            self.log_result("Workshop Repair Analysis", False, f"Error: {str(e)}")
            return False
    
    def test_finance_analyze_transactions(self):
        """Test POST /api/ai/finance/analyze-transactions?period_days=30"""
        print("\n=== TEST FINANCE TRANSACTION ANALYSIS ===")
        
        try:
            response = requests.post(f"{BASE_URL}/ai/finance/analyze-transactions?period_days=30", 
                                   headers=self.auth_headers)
            
            if response.status_code == 200:
                result = response.json()
                
                checks = []
                if "analysis" in result:
                    checks.append("✓ Analysis field present")
                    
                    analysis = result["analysis"]
                    if isinstance(analysis, dict):
                        if "message" in analysis:
                            checks.append(f"✓ Analysis message: {analysis['message']}")
                        elif "anomalies" in analysis or "risk_score" in analysis:
                            checks.append("✓ Financial analysis structured")
                        elif "raw_response" in analysis:
                            checks.append("✓ AI response received (unstructured)")
                        else:
                            checks.append("✗ No valid analysis data")
                else:
                    checks.append("✗ Missing analysis field")
                
                if "period_days" in result and result["period_days"] == 30:
                    checks.append("✓ Period parameter handled correctly")
                
                if "timestamp" in result:
                    checks.append("✓ Timestamp present")
                
                self.log_result("Finance Transaction Analysis", True, 
                              "Transaction analysis completed", 
                              "\n".join(checks))
                return True
            else:
                self.log_result("Finance Transaction Analysis", False, 
                              f"Failed: {response.status_code} - {response.text}")
                return False
                
        except Exception as e:
            self.log_result("Finance Transaction Analysis", False, f"Error: {str(e)}")
            return False
    
    def test_finance_predict_cashflow(self):
        """Test POST /api/ai/finance/predict-cashflow?months_ahead=3"""
        print("\n=== TEST FINANCE CASHFLOW PREDICTION ===")
        
        try:
            response = requests.post(f"{BASE_URL}/ai/finance/predict-cashflow?months_ahead=3", 
                                   headers=self.auth_headers)
            
            if response.status_code == 200:
                result = response.json()
                
                checks = []
                if "prediction" in result:
                    checks.append("✓ Prediction field present")
                    
                    prediction = result["prediction"]
                    if isinstance(prediction, dict):
                        if "message" in prediction:
                            checks.append(f"✓ Prediction message: {prediction['message']}")
                        elif "predictions" in prediction:
                            checks.append("✓ Cashflow predictions structured")
                        elif "raw_response" in prediction:
                            checks.append("✓ AI response received (unstructured)")
                        else:
                            checks.append("✗ No valid prediction data")
                else:
                    checks.append("✗ Missing prediction field")
                
                if "months_ahead" in result and result["months_ahead"] == 3:
                    checks.append("✓ Months parameter handled correctly")
                
                if "timestamp" in result:
                    checks.append("✓ Timestamp present")
                
                self.log_result("Finance Cashflow Prediction", True, 
                              "Cashflow prediction completed", 
                              "\n".join(checks))
                return True
            else:
                self.log_result("Finance Cashflow Prediction", False, 
                              f"Failed: {response.status_code} - {response.text}")
                return False
                
        except Exception as e:
            self.log_result("Finance Cashflow Prediction", False, f"Error: {str(e)}")
            return False
    
    def test_hr_analyze_absences(self):
        """Test POST /api/ai/hr/analyze-absences"""
        print("\n=== TEST HR ABSENCE ANALYSIS ===")
        
        try:
            response = requests.post(f"{BASE_URL}/ai/hr/analyze-absences", 
                                   headers=self.auth_headers)
            
            if response.status_code == 200:
                result = response.json()
                
                checks = []
                if "analysis" in result:
                    checks.append("✓ Analysis field present")
                    
                    analysis = result["analysis"]
                    if isinstance(analysis, dict):
                        if "message" in analysis:
                            checks.append(f"✓ Analysis message: {analysis['message']}")
                        elif "patterns" in analysis or "anomalies" in analysis:
                            checks.append("✓ HR absence analysis structured")
                        elif "raw_response" in analysis:
                            checks.append("✓ AI response received (unstructured)")
                        else:
                            checks.append("✗ No valid analysis data")
                else:
                    checks.append("✗ Missing analysis field")
                
                if "employees_analyzed" in result:
                    checks.append(f"✓ Employees analyzed: {result['employees_analyzed']}")
                
                if "timestamp" in result:
                    checks.append("✓ Timestamp present")
                
                self.log_result("HR Absence Analysis", True, 
                              "Absence analysis completed", 
                              "\n".join(checks))
                return True
            else:
                self.log_result("HR Absence Analysis", False, 
                              f"Failed: {response.status_code} - {response.text}")
                return False
                
        except Exception as e:
            self.log_result("HR Absence Analysis", False, f"Error: {str(e)}")
            return False
    
    def run_all_tests(self):
        """Run all AI module tests"""
        print("🤖 AI MODULE TESTING - GPT-5 VIA EMERGENT KEY")
        print("=" * 60)
        
        # Authentication
        if not self.authenticate():
            print("\n❌ Authentication failed. Cannot proceed with AI tests.")
            return False
        
        # Run all AI tests
        self.test_general_insights()
        self.test_workshop_detect_anomalies()
        self.test_workshop_analyze_repair()
        self.test_finance_analyze_transactions()
        self.test_finance_predict_cashflow()
        self.test_hr_analyze_absences()
        
        # Summary
        print("\n" + "=" * 60)
        print("📊 AI MODULE TEST SUMMARY")
        print("=" * 60)
        
        passed = sum(1 for result in self.test_results if result["success"])
        total = len(self.test_results)
        
        print(f"Total Tests: {total}")
        print(f"Passed: {passed}")
        print(f"Failed: {total - passed}")
        print(f"Success Rate: {(passed/total)*100:.1f}%")
        
        # Show failed tests
        failed_tests = [result for result in self.test_results if not result["success"]]
        if failed_tests:
            print("\n❌ FAILED TESTS:")
            for test in failed_tests:
                print(f"  - {test['test']}: {test['message']}")
        
        return passed == total

if __name__ == "__main__":
    print("🚀 STARTING ERP AI MODULE TESTING")
    print("=" * 60)
    
    # Test AI Module (New GPT-5 Integration)
    ai_tester = AITester()
    ai_success = ai_tester.run_all_tests()
    
    print("\n" + "=" * 60)
    print("🎯 AI MODULE TEST RESULTS")
    print("=" * 60)
    
    if ai_success:
        print("✅ AI MODULE TESTS PASSED - GPT-5 INTEGRATION OPERATIONAL!")
        sys.exit(0)
    else:
        print("❌ AI MODULE TESTS FAILED - CHECK RESULTS ABOVE")
        sys.exit(1)