from fastapi import FastAPI, APIRouter, HTTPException, status, Depends
from fastapi.security import HTTPBearer
from dotenv import load_dotenv
from starlette.middleware.cors import CORSMiddleware
from motor.motor_asyncio import AsyncIOMotorClient
import os
import logging
from pathlib import Path
from typing import List, Optional
from datetime import datetime, timezone

from models import (
    User, UserCreate, UserLogin, UserResponse, UserRole,
    Brand, BrandCreate,
    Category, CategoryCreate,
    Product, ProductCreate, ProductUpdate,
    Client, ClientCreate, ClientUpdate, ClientCategory,
    Order, OrderCreate, OrderStatus, OrderStatusUpdate,
    RepairOrder, RepairOrderCreate, RepairOrderUpdate, RepairOrderStatus, RepairOrderStatusUpdate,
    DashboardStats
)
from auth import get_password_hash, verify_password, create_access_token
from dependencies import get_current_user

# Import new routers
from routes_finance import router as finance_router
from routes_fiscal import router as fiscal_router
from routes_hr import router as hr_router
from routes_ged import router as ged_router
from routes_users import router as users_router
from routes_ai import router as ai_router

ROOT_DIR = Path(__file__).parent
load_dotenv(ROOT_DIR / '.env')

# MongoDB connection
mongo_url = os.environ['MONGO_URL']
client = AsyncIOMotorClient(mongo_url)
db = client[os.environ['DB_NAME']]

# Collections
users_col = db.users
brands_col = db.brands
categories_col = db.categories
products_col = db.products
clients_col = db.clients
orders_col = db.orders
repair_orders_col = db.repair_orders

# Create the main app
app = FastAPI(title="ERP Auto Algeria API")
api_router = APIRouter(prefix="/api")
security = HTTPBearer()

# ==================== AUTH ROUTES ====================
@api_router.post("/auth/register", response_model=UserResponse)
async def register(user_data: UserCreate, current_user: dict = Depends(get_current_user)):
    # Seul le SUPERADMIN peut créer des utilisateurs
    if current_user.get("role") != "superadmin":
        raise HTTPException(status_code=403, detail="Only superadmin can create users")
    
    # Check if user exists
    existing_user = await users_col.find_one({"email": user_data.email}, {"_id": 0})
    if existing_user:
        raise HTTPException(status_code=400, detail="Email already registered")
    
    # Create user
    user = User(
        email=user_data.email,
        password_hash=get_password_hash(user_data.password),
        full_name=user_data.full_name,
        phone=user_data.phone,
        role=user_data.role,
        language=user_data.language,
        permissions=user_data.permissions  # Inclure les permissions personnalisées
    )
    
    user_dict = user.model_dump()
    user_dict['created_at'] = user_dict['created_at'].isoformat()
    
    # Convertir les permissions en dict si elles existent
    if user_dict.get('permissions'):
        user_dict['permissions'] = user_dict['permissions']
    
    await users_col.insert_one(user_dict)
    
    return UserResponse(**user.model_dump())

@api_router.post("/auth/login")
async def login(credentials: UserLogin):
    user = await users_col.find_one({"email": credentials.email}, {"_id": 0})
    if not user or not verify_password(credentials.password, user["password_hash"]):
        raise HTTPException(status_code=401, detail="Invalid credentials")
    
    if not user.get("is_active", True):
        raise HTTPException(status_code=403, detail="Account is inactive")
    
    token = create_access_token(data={
        "user_id": user["id"],
        "email": user["email"],
        "role": user["role"]
    })
    
    return {
        "access_token": token,
        "token_type": "bearer",
        "user": UserResponse(**user)
    }

@api_router.get("/auth/me", response_model=UserResponse)
async def get_me(current_user: dict = Depends(get_current_user)):
    user = await users_col.find_one({"id": current_user["user_id"]}, {"_id": 0})
    if not user:
        raise HTTPException(status_code=404, detail="User not found")
    return UserResponse(**user)

# ==================== BRANDS ROUTES ====================
@api_router.get("/brands", response_model=List[Brand])
async def get_brands():
    brands = await brands_col.find({}, {"_id": 0}).to_list(1000)
    for brand in brands:
        if isinstance(brand.get('created_at'), str):
            brand['created_at'] = datetime.fromisoformat(brand['created_at'])
    return brands

@api_router.post("/brands", response_model=Brand)
async def create_brand(brand_data: BrandCreate, current_user: dict = Depends(get_current_user)):
    if current_user.get("role") not in ["superadmin", "admin"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    brand = Brand(**brand_data.model_dump())
    brand_dict = brand.model_dump()
    brand_dict['created_at'] = brand_dict['created_at'].isoformat()
    
    await brands_col.insert_one(brand_dict)
    return brand

@api_router.delete("/brands/{brand_id}")
async def delete_brand(brand_id: str, current_user: dict = Depends(get_current_user)):
    if current_user.get("role") not in ["superadmin", "admin"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    result = await brands_col.delete_one({"id": brand_id})
    if result.deleted_count == 0:
        raise HTTPException(status_code=404, detail="Brand not found")
    return {"message": "Brand deleted successfully"}

# ==================== CATEGORIES ROUTES ====================
@api_router.get("/categories", response_model=List[Category])
async def get_categories():
    categories = await categories_col.find({}, {"_id": 0}).to_list(1000)
    for category in categories:
        if isinstance(category.get('created_at'), str):
            category['created_at'] = datetime.fromisoformat(category['created_at'])
    return categories

@api_router.post("/categories", response_model=Category)
async def create_category(category_data: CategoryCreate, current_user: dict = Depends(get_current_user)):
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    category = Category(**category_data.model_dump())
    category_dict = category.model_dump()
    category_dict['created_at'] = category_dict['created_at'].isoformat()
    
    await categories_col.insert_one(category_dict)
    return category

@api_router.delete("/categories/{category_id}")
async def delete_category(category_id: str, current_user: dict = Depends(get_current_user)):
    if current_user.get("role") not in ["superadmin", "admin"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    result = await categories_col.delete_one({"id": category_id})
    if result.deleted_count == 0:
        raise HTTPException(status_code=404, detail="Category not found")
    return {"message": "Category deleted successfully"}

# ==================== PRODUCTS ROUTES ====================
@api_router.get("/products", response_model=List[Product])
async def get_products(
    search: Optional[str] = None,
    category_id: Optional[str] = None,
    brand_id: Optional[str] = None,
    skip: int = 0,
    limit: int = 100
):
    query = {"is_active": True}
    
    if search:
        query["$or"] = [
            {"reference": {"$regex": search, "$options": "i"}},
            {"name": {"$regex": search, "$options": "i"}},
            {"oem_references": {"$regex": search, "$options": "i"}}
        ]
    
    if category_id:
        query["category_id"] = category_id
    
    if brand_id:
        query["brand_id"] = brand_id
    
    products = await products_col.find(query, {"_id": 0}).skip(skip).limit(limit).to_list(limit)
    
    for product in products:
        if isinstance(product.get('created_at'), str):
            product['created_at'] = datetime.fromisoformat(product['created_at'])
    
    return products

@api_router.get("/products/{product_id}", response_model=Product)
async def get_product(product_id: str):
    product = await products_col.find_one({"id": product_id}, {"_id": 0})
    if not product:
        raise HTTPException(status_code=404, detail="Product not found")
    
    if isinstance(product.get('created_at'), str):
        product['created_at'] = datetime.fromisoformat(product['created_at'])
    
    return product

@api_router.post("/products", response_model=Product)
async def create_product(product_data: ProductCreate, current_user: dict = Depends(get_current_user)):
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    # Check if reference exists
    existing = await products_col.find_one({"reference": product_data.reference}, {"_id": 0})
    if existing:
        raise HTTPException(status_code=400, detail="Product reference already exists")
    
    product = Product(**product_data.model_dump())
    product_dict = product.model_dump()
    product_dict['created_at'] = product_dict['created_at'].isoformat()
    
    await products_col.insert_one(product_dict)
    return product

@api_router.put("/products/{product_id}", response_model=Product)
async def update_product(product_id: str, product_data: ProductUpdate, current_user: dict = Depends(get_current_user)):
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    existing = await products_col.find_one({"id": product_id}, {"_id": 0})
    if not existing:
        raise HTTPException(status_code=404, detail="Product not found")
    
    update_data = {k: v for k, v in product_data.model_dump().items() if v is not None}
    update_data["updated_at"] = datetime.now(timezone.utc).isoformat()
    
    await products_col.update_one({"id": product_id}, {"$set": update_data})
    
    updated_product = await products_col.find_one({"id": product_id}, {"_id": 0})
    if isinstance(updated_product.get('created_at'), str):
        updated_product['created_at'] = datetime.fromisoformat(updated_product['created_at'])
    
    return updated_product

@api_router.delete("/products/{product_id}")
async def delete_product(product_id: str, current_user: dict = Depends(get_current_user)):
    if current_user.get("role") not in ["superadmin", "admin"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    # Soft delete
    await products_col.update_one({"id": product_id}, {"$set": {"is_active": False}})
    return {"message": "Product deleted successfully"}

# ==================== CLIENTS ROUTES ====================
@api_router.get("/clients", response_model=List[Client])
async def get_clients(current_user: dict = Depends(get_current_user)):
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    clients = await clients_col.find({}, {"_id": 0}).to_list(1000)
    for client in clients:
        if isinstance(client.get('created_at'), str):
            client['created_at'] = datetime.fromisoformat(client['created_at'])
    return clients

@api_router.get("/clients/{client_id}", response_model=Client)
async def get_client(client_id: str, current_user: dict = Depends(get_current_user)):
    client = await clients_col.find_one({"id": client_id}, {"_id": 0})
    if not client:
        raise HTTPException(status_code=404, detail="Client not found")
    
    if isinstance(client.get('created_at'), str):
        client['created_at'] = datetime.fromisoformat(client['created_at'])
    
    return client

@api_router.post("/clients", response_model=Client)
async def create_client(client_data: ClientCreate, current_user: dict = Depends(get_current_user)):
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    client = Client(**client_data.model_dump())
    client_dict = client.model_dump()
    client_dict['created_at'] = client_dict['created_at'].isoformat()
    
    await clients_col.insert_one(client_dict)
    return client

@api_router.put("/clients/{client_id}", response_model=Client)
async def update_client(client_id: str, client_data: ClientUpdate, current_user: dict = Depends(get_current_user)):
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    update_data = {k: v for k, v in client_data.model_dump().items() if v is not None}
    update_data["updated_at"] = datetime.now(timezone.utc).isoformat()
    
    await clients_col.update_one({"id": client_id}, {"$set": update_data})
    
    updated_client = await clients_col.find_one({"id": client_id}, {"_id": 0})
    if isinstance(updated_client.get('created_at'), str):
        updated_client['created_at'] = datetime.fromisoformat(updated_client['created_at'])
    
    return updated_client

# ==================== ORDERS ROUTES ====================
@api_router.get("/orders", response_model=List[Order])
async def get_orders(current_user: dict = Depends(get_current_user)):
    query = {}
    
    # If client role, only show their orders
    if current_user.get("role") == "client":
        # Get client_id from user_id
        client = await clients_col.find_one({"user_id": current_user["user_id"]}, {"_id": 0})
        if client:
            query["client_id"] = client["id"]
    
    orders = await orders_col.find(query, {"_id": 0}).sort("created_at", -1).to_list(1000)
    
    for order in orders:
        if isinstance(order.get('created_at'), str):
            order['created_at'] = datetime.fromisoformat(order['created_at'])
    
    return orders

@api_router.get("/orders/{order_id}", response_model=Order)
async def get_order(order_id: str, current_user: dict = Depends(get_current_user)):
    order = await orders_col.find_one({"id": order_id}, {"_id": 0})
    if not order:
        raise HTTPException(status_code=404, detail="Order not found")
    
    if isinstance(order.get('created_at'), str):
        order['created_at'] = datetime.fromisoformat(order['created_at'])
    
    return order

@api_router.post("/orders", response_model=Order)
async def create_order(order_data: OrderCreate, current_user: dict = Depends(get_current_user)):
    # Get client info
    client = await clients_col.find_one({"id": order_data.client_id}, {"_id": 0})
    if not client:
        raise HTTPException(status_code=404, detail="Client not found")
    
    # Calculate totals
    subtotal = sum(item.subtotal for item in order_data.items)
    tax_amount = subtotal * 0.19  # TVA 19%
    total = subtotal + tax_amount
    
    # Generate order number
    order_count = await orders_col.count_documents({})
    order_number = f"CMD-{order_count + 1:06d}"
    
    order = Order(
        order_number=order_number,
        client_id=order_data.client_id,
        client_name=client["company_name"],
        items=order_data.items,
        subtotal=subtotal,
        tax_amount=tax_amount,
        total=total,
        notes=order_data.notes,
        delivery_address=order_data.delivery_address or client.get("address")
    )
    
    order_dict = order.model_dump()
    order_dict['created_at'] = order_dict['created_at'].isoformat()
    
    await orders_col.insert_one(order_dict)
    
    # Update stock
    for item in order_data.items:
        await products_col.update_one(
            {"id": item.product_id},
            {"$inc": {"stock_qty": -item.quantity}}
        )
    
    return order

@api_router.patch("/orders/{order_id}/status")
async def update_order_status(
    order_id: str,
    status_update: OrderStatusUpdate,
    current_user: dict = Depends(get_current_user)
):
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    result = await orders_col.update_one(
        {"id": order_id},
        {"$set": {"status": status_update.status.value, "updated_at": datetime.now(timezone.utc).isoformat()}}
    )
    
    if result.modified_count == 0:
        raise HTTPException(status_code=404, detail="Order not found")
    
    return {"message": "Order status updated successfully"}

# ==================== DASHBOARD ROUTES ====================
@api_router.get("/dashboard/stats", response_model=DashboardStats)
async def get_dashboard_stats(current_user: dict = Depends(get_current_user)):
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    total_products = await products_col.count_documents({"is_active": True})
    low_stock_products = await products_col.count_documents({"$expr": {"$lt": ["$stock_qty", "$min_stock"]}})
    total_clients = await clients_col.count_documents({})
    pending_orders = await orders_col.count_documents({"status": "pending"})
    
    # Calculate total revenue
    orders = await orders_col.find({"status": {"$in": ["delivered"]}}, {"_id": 0, "total": 1}).to_list(10000)
    total_revenue = sum(order.get("total", 0) for order in orders)
    
    # Get recent orders
    recent_orders_cursor = orders_col.find({}, {"_id": 0}).sort("created_at", -1).limit(10)
    recent_orders = await recent_orders_cursor.to_list(10)
    
    return DashboardStats(
        total_products=total_products,
        low_stock_products=low_stock_products,
        total_clients=total_clients,
        pending_orders=pending_orders,
        total_revenue=total_revenue,
        recent_orders=recent_orders
    )

# ==================== REPAIR ORDERS / WORKSHOP ROUTES ====================
@api_router.get("/repair-orders", response_model=List[RepairOrder])
async def get_repair_orders(
    current_user: dict = Depends(get_current_user),
    status: Optional[str] = None,
    search: Optional[str] = None,
    skip: int = 0,
    limit: int = 100
):
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    query = {}
    
    if status:
        query["status"] = status
    
    if search:
        query["$or"] = [
            {"repair_number": {"$regex": search, "$options": "i"}},
            {"client_name": {"$regex": search, "$options": "i"}},
            {"vehicle_plate": {"$regex": search, "$options": "i"}},
            {"vehicle_vin": {"$regex": search, "$options": "i"}}
        ]
    
    repair_orders = await repair_orders_col.find(query, {"_id": 0}).sort("created_at", -1).skip(skip).limit(limit).to_list(limit)
    
    for repair_order in repair_orders:
        if isinstance(repair_order.get('created_at'), str):
            repair_order['created_at'] = datetime.fromisoformat(repair_order['created_at'])
        if isinstance(repair_order.get('scheduled_date'), str):
            repair_order['scheduled_date'] = datetime.fromisoformat(repair_order['scheduled_date'])
        if isinstance(repair_order.get('completion_date'), str):
            repair_order['completion_date'] = datetime.fromisoformat(repair_order['completion_date'])
    
    return repair_orders

@api_router.get("/repair-orders/{repair_order_id}", response_model=RepairOrder)
async def get_repair_order(repair_order_id: str, current_user: dict = Depends(get_current_user)):
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    repair_order = await repair_orders_col.find_one({"id": repair_order_id}, {"_id": 0})
    if not repair_order:
        raise HTTPException(status_code=404, detail="Repair order not found")
    
    if isinstance(repair_order.get('created_at'), str):
        repair_order['created_at'] = datetime.fromisoformat(repair_order['created_at'])
    if isinstance(repair_order.get('scheduled_date'), str):
        repair_order['scheduled_date'] = datetime.fromisoformat(repair_order['scheduled_date'])
    if isinstance(repair_order.get('completion_date'), str):
        repair_order['completion_date'] = datetime.fromisoformat(repair_order['completion_date'])
    
    return repair_order

@api_router.post("/repair-orders", response_model=RepairOrder)
async def create_repair_order(repair_order_data: RepairOrderCreate, current_user: dict = Depends(get_current_user)):
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    # Calculate totals
    labor_subtotal = sum(service.labor_cost for service in repair_order_data.services)
    parts_subtotal = sum(part.subtotal for part in repair_order_data.parts)
    subtotal = labor_subtotal + parts_subtotal
    tax_amount = subtotal * 0.19  # TVA 19%
    total = subtotal + tax_amount
    
    # Generate repair number
    repair_count = await repair_orders_col.count_documents({})
    repair_number = f"REP-{repair_count + 1:06d}"
    
    repair_order = RepairOrder(
        repair_number=repair_number,
        client_id=repair_order_data.client_id,
        client_name=repair_order_data.client_name,
        vehicle_make=repair_order_data.vehicle_make,
        vehicle_model=repair_order_data.vehicle_model,
        vehicle_year=repair_order_data.vehicle_year,
        vehicle_vin=repair_order_data.vehicle_vin,
        vehicle_plate=repair_order_data.vehicle_plate,
        mileage=repair_order_data.mileage,
        priority=repair_order_data.priority,
        scheduled_date=repair_order_data.scheduled_date,
        assigned_technician=repair_order_data.assigned_technician,
        description=repair_order_data.description,
        diagnosis=repair_order_data.diagnosis,
        services=repair_order_data.services,
        parts=repair_order_data.parts,
        labor_subtotal=labor_subtotal,
        parts_subtotal=parts_subtotal,
        subtotal=subtotal,
        tax_amount=tax_amount,
        total=total,
        notes=repair_order_data.notes
    )
    
    repair_order_dict = repair_order.model_dump()
    repair_order_dict['created_at'] = repair_order_dict['created_at'].isoformat()
    if repair_order_dict.get('scheduled_date'):
        repair_order_dict['scheduled_date'] = repair_order_dict['scheduled_date'].isoformat()
    
    await repair_orders_col.insert_one(repair_order_dict)
    
    # Update stock for parts used
    for part in repair_order_data.parts:
        await products_col.update_one(
            {"id": part.product_id},
            {"$inc": {"stock_qty": -part.quantity}}
        )
    
    return repair_order

@api_router.put("/repair-orders/{repair_order_id}", response_model=RepairOrder)
async def update_repair_order(
    repair_order_id: str,
    repair_order_data: RepairOrderUpdate,
    current_user: dict = Depends(get_current_user)
):
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    existing = await repair_orders_col.find_one({"id": repair_order_id}, {"_id": 0})
    if not existing:
        raise HTTPException(status_code=404, detail="Repair order not found")
    
    update_data = {k: v for k, v in repair_order_data.model_dump().items() if v is not None}
    
    # Recalculate totals if services or parts are updated
    if "services" in update_data or "parts" in update_data:
        services = update_data.get("services", existing.get("services", []))
        parts = update_data.get("parts", existing.get("parts", []))
        
        labor_subtotal = sum(service["labor_cost"] for service in services)
        parts_subtotal = sum(part["subtotal"] for part in parts)
        subtotal = labor_subtotal + parts_subtotal
        tax_amount = subtotal * 0.19
        total = subtotal + tax_amount
        
        update_data["labor_subtotal"] = labor_subtotal
        update_data["parts_subtotal"] = parts_subtotal
        update_data["subtotal"] = subtotal
        update_data["tax_amount"] = tax_amount
        update_data["total"] = total
    
    update_data["updated_at"] = datetime.now(timezone.utc).isoformat()
    
    # Convert datetime objects to ISO strings
    if "scheduled_date" in update_data and update_data["scheduled_date"]:
        update_data["scheduled_date"] = update_data["scheduled_date"].isoformat()
    if "completion_date" in update_data and update_data["completion_date"]:
        update_data["completion_date"] = update_data["completion_date"].isoformat()
    
    await repair_orders_col.update_one({"id": repair_order_id}, {"$set": update_data})
    
    updated_repair_order = await repair_orders_col.find_one({"id": repair_order_id}, {"_id": 0})
    if isinstance(updated_repair_order.get('created_at'), str):
        updated_repair_order['created_at'] = datetime.fromisoformat(updated_repair_order['created_at'])
    if isinstance(updated_repair_order.get('scheduled_date'), str):
        updated_repair_order['scheduled_date'] = datetime.fromisoformat(updated_repair_order['scheduled_date'])
    if isinstance(updated_repair_order.get('completion_date'), str):
        updated_repair_order['completion_date'] = datetime.fromisoformat(updated_repair_order['completion_date'])
    
    return updated_repair_order

@api_router.patch("/repair-orders/{repair_order_id}/status")
async def update_repair_order_status(
    repair_order_id: str,
    status_update: RepairOrderStatusUpdate,
    current_user: dict = Depends(get_current_user)
):
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    update_data = {"status": status_update.status.value, "updated_at": datetime.now(timezone.utc).isoformat()}
    
    # Set completion date when status is completed
    if status_update.status == RepairOrderStatus.COMPLETED:
        update_data["completion_date"] = datetime.now(timezone.utc).isoformat()
    
    result = await repair_orders_col.update_one(
        {"id": repair_order_id},
        {"$set": update_data}
    )
    
    if result.modified_count == 0:
        raise HTTPException(status_code=404, detail="Repair order not found")
    
    return {"message": "Repair order status updated successfully"}

@api_router.delete("/repair-orders/{repair_order_id}")
async def delete_repair_order(repair_order_id: str, current_user: dict = Depends(get_current_user)):
    if current_user.get("role") not in ["superadmin", "admin"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    result = await repair_orders_col.delete_one({"id": repair_order_id})
    if result.deleted_count == 0:
        raise HTTPException(status_code=404, detail="Repair order not found")
    return {"message": "Repair order deleted successfully"}

# ==================== USERS MANAGEMENT ====================
# Users management is now handled by routes_users.py (imported as users_router)

# Include the router in the main app
app.include_router(api_router)

# Include new module routers with /api prefix
app.include_router(finance_router, prefix="/api")
app.include_router(fiscal_router, prefix="/api")
app.include_router(hr_router, prefix="/api")
app.include_router(ged_router, prefix="/api")
app.include_router(users_router, prefix="/api")
app.include_router(ai_router, prefix="/api")

app.add_middleware(
    CORSMiddleware,
    allow_credentials=True,
    allow_origins=os.environ.get('CORS_ORIGINS', '*').split(','),
    allow_methods=["*"],
    allow_headers=["*"],
)

# Configure logging
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
)
logger = logging.getLogger(__name__)

@app.on_event("shutdown")
async def shutdown_db_client():
    client.close()