"""
Script to seed initial data for development
Run: python seed_data.py
"""
import asyncio
from motor.motor_asyncio import AsyncIOMotorClient
from auth import get_password_hash
from models import UserRole, ClientCategory
import os
from dotenv import load_dotenv
from pathlib import Path

ROOT_DIR = Path(__file__).parent
load_dotenv(ROOT_DIR / '.env')

mongo_url = os.environ['MONGO_URL']
client = AsyncIOMotorClient(mongo_url)
db = client[os.environ['DB_NAME']]

async def seed_data():
    print("🌱 Seeding database...")
    
    # Clear existing data
    await db.users.delete_many({})
    await db.brands.delete_many({})
    await db.categories.delete_many({})
    await db.products.delete_many({})
    await db.clients.delete_many({})
    await db.orders.delete_many({})
    
    # Create Admin User
    admin_user = {
        "id": "admin-001",
        "email": "admin@erpauto.dz",
        "password_hash": get_password_hash("admin123"),
        "full_name": "Administrateur Principal",
        "phone": "+213555000000",
        "role": "superadmin",
        "is_active": True,
        "language": "fr",
        "created_at": "2025-01-01T00:00:00Z"
    }
    await db.users.insert_one(admin_user)
    print("✅ Admin user created: admin@erpauto.dz / admin123")
    
    # Create Client User
    client_user = {
        "id": "client-001",
        "email": "client@garage.dz",
        "password_hash": get_password_hash("client123"),
        "full_name": "Mohamed Garage SARL",
        "phone": "+213666111222",
        "role": "client",
        "is_active": True,
        "language": "fr",
        "created_at": "2025-01-01T00:00:00Z"
    }
    await db.users.insert_one(client_user)
    print("✅ Client user created: client@garage.dz / client123")
    
    # Create Brands
    brands = [
        {"id": "brand-001", "name": "Bosch", "country": "Allemagne", "logo": "https://via.placeholder.com/100?text=Bosch", "created_at": "2025-01-01T00:00:00Z"},
        {"id": "brand-002", "name": "Mann Filter", "country": "Allemagne", "logo": "https://via.placeholder.com/100?text=Mann", "created_at": "2025-01-01T00:00:00Z"},
        {"id": "brand-003", "name": "NGK", "country": "Japon", "logo": "https://via.placeholder.com/100?text=NGK", "created_at": "2025-01-01T00:00:00Z"},
        {"id": "brand-004", "name": "Valeo", "country": "France", "logo": "https://via.placeholder.com/100?text=Valeo", "created_at": "2025-01-01T00:00:00Z"},
    ]
    await db.brands.insert_many(brands)
    print(f"✅ {len(brands)} brands created")
    
    # Create Categories
    categories = [
        {"id": "cat-001", "name": "Filtres", "icon": "filter", "created_at": "2025-01-01T00:00:00Z"},
        {"id": "cat-002", "name": "Freinage", "icon": "disc-brake", "created_at": "2025-01-01T00:00:00Z"},
        {"id": "cat-003", "name": "Allumage", "icon": "spark", "created_at": "2025-01-01T00:00:00Z"},
        {"id": "cat-004", "name": "Éclairage", "icon": "lightbulb", "created_at": "2025-01-01T00:00:00Z"},
        {"id": "cat-005", "name": "Moteur", "icon": "engine", "created_at": "2025-01-01T00:00:00Z"},
    ]
    await db.categories.insert_many(categories)
    print(f"✅ {len(categories)} categories created")
    
    # Create Products
    products = [
        {
            "id": "prod-001",
            "reference": "FO-001",
            "name": "Filtre à huile Bosch",
            "description": "Filtre à huile haute qualité pour moteurs essence et diesel",
            "category_id": "cat-001",
            "brand_id": "brand-001",
            "stock_qty": 150,
            "min_stock": 20,
            "prices": {"public": 800, "garage": 650, "semi_wholesale": 550, "wholesale": 450},
            "images": ["https://via.placeholder.com/300?text=Filtre+Huile"],
            "specifications": {"Type": "Vissable", "Diamètre": "76mm"},
            "is_active": True,
            "oem_references": ["OE-001", "OE-002"],
            "created_at": "2025-01-01T00:00:00Z"
        },
        {
            "id": "prod-002",
            "reference": "PQ-100",
            "name": "Plaquettes de frein avant",
            "description": "Plaquettes de frein céramiques haute performance",
            "category_id": "cat-002",
            "brand_id": "brand-001",
            "stock_qty": 80,
            "min_stock": 15,
            "prices": {"public": 4500, "garage": 3800, "semi_wholesale": 3400, "wholesale": 3000},
            "images": ["https://via.placeholder.com/300?text=Plaquettes"],
            "specifications": {"Position": "Avant", "Matériau": "Céramique"},
            "is_active": True,
            "oem_references": ["BP-001"],
            "created_at": "2025-01-01T00:00:00Z"
        },
        {
            "id": "prod-003",
            "reference": "BP-200",
            "name": "Bougie d'allumage NGK",
            "description": "Bougie standard pour véhicules essence",
            "category_id": "cat-003",
            "brand_id": "brand-003",
            "stock_qty": 5,
            "min_stock": 50,
            "prices": {"public": 450, "garage": 350, "semi_wholesale": 300, "wholesale": 250},
            "images": ["https://via.placeholder.com/300?text=Bougie"],
            "specifications": {"Écart": "0.8mm", "Type": "Standard"},
            "is_active": True,
            "oem_references": ["SP-100"],
            "created_at": "2025-01-01T00:00:00Z"
        },
        {
            "id": "prod-004",
            "reference": "AM-300",
            "name": "Ampoule H7 Valeo",
            "description": "Ampoule halogène H7 12V 55W",
            "category_id": "cat-004",
            "brand_id": "brand-004",
            "stock_qty": 200,
            "min_stock": 30,
            "prices": {"public": 350, "garage": 280, "semi_wholesale": 240, "wholesale": 200},
            "images": ["https://via.placeholder.com/300?text=Ampoule+H7"],
            "specifications": {"Voltage": "12V", "Puissance": "55W"},
            "is_active": True,
            "oem_references": [],
            "created_at": "2025-01-01T00:00:00Z"
        },
        {
            "id": "prod-005",
            "reference": "FA-401",
            "name": "Filtre à air Mann",
            "description": "Filtre à air haute filtration",
            "category_id": "cat-001",
            "brand_id": "brand-002",
            "stock_qty": 120,
            "min_stock": 25,
            "prices": {"public": 1200, "garage": 950, "semi_wholesale": 850, "wholesale": 750},
            "images": ["https://via.placeholder.com/300?text=Filtre+Air"],
            "specifications": {"Type": "Papier", "Dimension": "Standard"},
            "is_active": True,
            "oem_references": ["AF-200"],
            "created_at": "2025-01-01T00:00:00Z"
        }
    ]
    await db.products.insert_many(products)
    print(f"✅ {len(products)} products created")
    
    # Create Client Profile
    client_profile = {
        "id": "client-profile-001",
        "user_id": "client-001",
        "company_name": "Garage Mohamed SARL",
        "category": "garage",
        "credit_limit": 500000.0,
        "payment_terms": 30,
        "address": "Zone Industrielle, Alger",
        "city": "Alger",
        "tax_id": "099999999999999",
        "commercial_register": "16/00-9999999",
        "balance": 0.0,
        "created_at": "2025-01-01T00:00:00Z"
    }
    await db.clients.insert_one(client_profile)
    print("✅ Client profile created")
    
    print("\n✨ Database seeded successfully!")
    print("\n📝 Test Accounts:")
    print("   Admin: admin@erpauto.dz / admin123")
    print("   Client: client@garage.dz / client123")
    
    client.close()

if __name__ == "__main__":
    asyncio.run(seed_data())
