from fastapi import APIRouter, HTTPException, Depends, UploadFile, File
from typing import List, Optional
from datetime import datetime, timezone
from models import (
    Document, DocumentCreate, DocumentCategory, DocumentStatus, DocumentApproval
)
from dependencies import get_current_user, get_db
from motor.motor_asyncio import AsyncIOMotorDatabase
import os
import shutil

router = APIRouter(prefix="/documents", tags=["documents"])

# Documents storage path
DOCUMENTS_PATH = "/app/documents"
os.makedirs(DOCUMENTS_PATH, exist_ok=True)

# ==================== DOCUMENTS ====================
@router.get("", response_model=List[Document])
async def get_documents(
    category: Optional[str] = None,
    status: Optional[str] = None,
    tag: Optional[str] = None,
    search: Optional[str] = None,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    query = {}
    
    if category:
        query["category"] = category
    if status:
        query["status"] = status
    if tag:
        query["tags"] = tag
    if search:
        query["$or"] = [
            {"name": {"$regex": search, "$options": "i"}},
            {"description": {"$regex": search, "$options": "i"}},
            {"reference": {"$regex": search, "$options": "i"}}
        ]
    
    documents = await db.documents.find(query, {"_id": 0}).sort("created_at", -1).to_list(None)
    
    for doc in documents:
        if isinstance(doc.get('created_at'), str):
            doc['created_at'] = datetime.fromisoformat(doc['created_at'])
        if doc.get('date') and isinstance(doc['date'], str):
            doc['date'] = datetime.fromisoformat(doc['date'])
        if doc.get('approval_date') and isinstance(doc['approval_date'], str):
            doc['approval_date'] = datetime.fromisoformat(doc['approval_date'])
        if doc.get('archive_date') and isinstance(doc['archive_date'], str):
            doc['archive_date'] = datetime.fromisoformat(doc['archive_date'])
    
    return documents

@router.get("/{document_id}", response_model=Document)
async def get_document(
    document_id: str,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    document = await db.documents.find_one({"id": document_id}, {"_id": 0})
    if not document:
        raise HTTPException(status_code=404, detail="Document not found")
    
    if isinstance(document.get('created_at'), str):
        document['created_at'] = datetime.fromisoformat(document['created_at'])
    if document.get('date') and isinstance(document['date'], str):
        document['date'] = datetime.fromisoformat(document['date'])
    
    return document

@router.post("/upload", response_model=Document)
async def upload_document(
    file: UploadFile = File(...),
    name: Optional[str] = None,
    category: str = DocumentCategory.ADMINISTRATIVE.value,
    description: Optional[str] = None,
    tags: str = "",  # Comma-separated tags
    reference: Optional[str] = None,
    requires_approval: bool = False,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    # Create category folder
    category_path = os.path.join(DOCUMENTS_PATH, category)
    os.makedirs(category_path, exist_ok=True)
    
    # Generate unique filename
    timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
    file_extension = os.path.splitext(file.filename)[1]
    filename = f"{timestamp}_{file.filename}"
    file_path = os.path.join(category_path, filename)
    
    # Save file
    with open(file_path, "wb") as buffer:
        shutil.copyfileobj(file.file, buffer)
    
    # Get file size
    file_size = os.path.getsize(file_path)
    
    # Parse tags
    tag_list = [tag.strip() for tag in tags.split(",") if tag.strip()]
    
    # Create document record
    document_data = DocumentCreate(
        name=name or file.filename,
        category=DocumentCategory(category),
        description=description,
        file_path=file_path,
        file_type=file_extension.replace(".", ""),
        file_size=file_size,
        tags=tag_list,
        reference=reference,
        requires_approval=requires_approval
    )
    
    document = Document(
        **document_data.model_dump(),
        uploaded_by=current_user.get("id")
    )
    
    document_dict = document.model_dump()
    document_dict['created_at'] = document_dict['created_at'].isoformat()
    if document_dict.get('date'):
        document_dict['date'] = document_dict['date'].isoformat()
    
    await db.documents.insert_one(document_dict)
    
    return document

@router.delete("/{document_id}")
async def delete_document(
    document_id: str,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    if current_user.get("role") not in ["superadmin", "admin"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    # Get document to delete file
    document = await db.documents.find_one({"id": document_id})
    if not document:
        raise HTTPException(status_code=404, detail="Document not found")
    
    # Delete file from filesystem
    if os.path.exists(document["file_path"]):
        os.remove(document["file_path"])
    
    # Delete from database
    await db.documents.delete_one({"id": document_id})
    
    return {"message": "Document deleted successfully"}

@router.get("/{document_id}/download")
async def download_document(
    document_id: str,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    from fastapi.responses import FileResponse
    
    document = await db.documents.find_one({"id": document_id})
    if not document:
        raise HTTPException(status_code=404, detail="Document not found")
    
    file_path = document["file_path"]
    if not os.path.exists(file_path):
        raise HTTPException(status_code=404, detail="File not found on server")
    
    return FileResponse(
        path=file_path,
        filename=document["name"],
        media_type="application/octet-stream"
    )

# ==================== WORKFLOW APPROBATION ====================
@router.patch("/{document_id}/submit-for-approval")
async def submit_for_approval(
    document_id: str,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    result = await db.documents.update_one(
        {"id": document_id},
        {
            "$set": {
                "status": DocumentStatus.PENDING_APPROVAL.value,
                "updated_at": datetime.now(timezone.utc).isoformat()
            }
        }
    )
    
    if result.modified_count == 0:
        raise HTTPException(status_code=404, detail="Document not found")
    
    return {"message": "Document submitted for approval"}

@router.patch("/{document_id}/approve")
async def approve_document(
    document_id: str,
    approval_data: DocumentApproval,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    if current_user.get("role") not in ["superadmin", "admin", "manager"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    if approval_data.approved:
        update_data = {
            "status": DocumentStatus.APPROVED.value,
            "approved_by": current_user.get("id"),
            "approval_date": datetime.now(timezone.utc).isoformat(),
            "updated_at": datetime.now(timezone.utc).isoformat()
        }
    else:
        update_data = {
            "status": DocumentStatus.REJECTED.value,
            "approved_by": current_user.get("id"),
            "approval_date": datetime.now(timezone.utc).isoformat(),
            "rejection_reason": approval_data.rejection_reason,
            "updated_at": datetime.now(timezone.utc).isoformat()
        }
    
    result = await db.documents.update_one(
        {"id": document_id},
        {"$set": update_data}
    )
    
    if result.modified_count == 0:
        raise HTTPException(status_code=404, detail="Document not found")
    
    status = "approved" if approval_data.approved else "rejected"
    return {"message": f"Document {status} successfully"}

@router.patch("/{document_id}/archive")
async def archive_document(
    document_id: str,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    if current_user.get("role") not in ["superadmin", "admin"]:
        raise HTTPException(status_code=403, detail="Insufficient permissions")
    
    result = await db.documents.update_one(
        {"id": document_id},
        {
            "$set": {
                "status": DocumentStatus.ARCHIVED.value,
                "archive_date": datetime.now(timezone.utc).isoformat(),
                "updated_at": datetime.now(timezone.utc).isoformat()
            }
        }
    )
    
    if result.modified_count == 0:
        raise HTTPException(status_code=404, detail="Document not found")
    
    return {"message": "Document archived successfully"}

# ==================== VERSIONING ====================
@router.post("/{document_id}/new-version")
async def create_document_version(
    document_id: str,
    file: UploadFile = File(...),
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    # Get parent document
    parent = await db.documents.find_one({"id": document_id})
    if not parent:
        raise HTTPException(status_code=404, detail="Parent document not found")
    
    # Create new version
    category_path = os.path.join(DOCUMENTS_PATH, parent["category"])
    timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
    file_extension = os.path.splitext(file.filename)[1]
    filename = f"{timestamp}_v{parent['version'] + 1}_{file.filename}"
    file_path = os.path.join(category_path, filename)
    
    # Save file
    with open(file_path, "wb") as buffer:
        shutil.copyfileobj(file.file, buffer)
    
    file_size = os.path.getsize(file_path)
    
    # Create new document version
    new_doc = Document(
        name=parent["name"],
        category=DocumentCategory(parent["category"]),
        description=parent.get("description"),
        file_path=file_path,
        file_type=file_extension.replace(".", ""),
        file_size=file_size,
        tags=parent.get("tags", []),
        reference=parent.get("reference"),
        uploaded_by=current_user.get("id"),
        version=parent["version"] + 1,
        parent_document_id=document_id,
        status=DocumentStatus.DRAFT,
        requires_approval=parent.get("requires_approval", False)
    )
    
    new_doc_dict = new_doc.model_dump()
    new_doc_dict['created_at'] = new_doc_dict['created_at'].isoformat()
    
    await db.documents.insert_one(new_doc_dict)
    
    return new_doc

@router.get("/{document_id}/versions", response_model=List[Document])
async def get_document_versions(
    document_id: str,
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    # Get all versions of this document
    documents = await db.documents.find(
        {
            "$or": [
                {"id": document_id},
                {"parent_document_id": document_id}
            ]
        },
        {"_id": 0}
    ).sort("version", 1).to_list(None)
    
    for doc in documents:
        if isinstance(doc.get('created_at'), str):
            doc['created_at'] = datetime.fromisoformat(doc['created_at'])
    
    return documents

# ==================== STATISTIQUES ====================
@router.get("/stats/by-category")
async def get_stats_by_category(
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    pipeline = [
        {"$group": {
            "_id": "$category",
            "count": {"$sum": 1},
            "total_size": {"$sum": "$file_size"}
        }}
    ]
    
    stats = await db.documents.aggregate(pipeline).to_list(None)
    
    return {
        "by_category": [
            {
                "category": stat["_id"],
                "count": stat["count"],
                "total_size_mb": round(stat["total_size"] / (1024 * 1024), 2)
            }
            for stat in stats
        ]
    }

@router.get("/stats/dashboard")
async def get_documents_dashboard(
    current_user: dict = Depends(get_current_user),
    db: AsyncIOMotorDatabase = Depends(get_db)
):
    # Total documents
    total_documents = await db.documents.count_documents({})
    
    # By status
    pending_approval = await db.documents.count_documents({"status": DocumentStatus.PENDING_APPROVAL.value})
    approved = await db.documents.count_documents({"status": DocumentStatus.APPROVED.value})
    archived = await db.documents.count_documents({"status": DocumentStatus.ARCHIVED.value})
    
    # By category
    categories = await db.documents.distinct("category")
    
    # Total storage used
    pipeline = [
        {"$group": {
            "_id": None,
            "total_size": {"$sum": "$file_size"}
        }}
    ]
    storage_result = await db.documents.aggregate(pipeline).to_list(None)
    total_storage_mb = round(storage_result[0]["total_size"] / (1024 * 1024), 2) if storage_result else 0
    
    return {
        "total_documents": total_documents,
        "pending_approval": pending_approval,
        "approved": approved,
        "archived": archived,
        "categories_count": len(categories),
        "total_storage_mb": total_storage_mb
    }
