"""
Système de gestion des permissions
"""
from models import UserPermissions, ModulePermission, UserRole
from typing import Optional

# Permissions par défaut selon les rôles
def get_default_permissions(role: UserRole) -> UserPermissions:
    """Retourne les permissions par défaut pour un rôle donné"""
    
    # SUPERADMIN - Accès total à tout
    if role == UserRole.SUPERADMIN:
        full_access = ModulePermission(view=True, create=True, edit=True, delete=True, approve=True)
        return UserPermissions(
            dashboard=full_access,
            products=full_access,
            stock=full_access,
            orders=full_access,
            clients=full_access,
            workshop=full_access,
            finance=full_access,
            fiscal=full_access,
            hr=full_access,
            ged=full_access,
            reports=full_access,
            users=full_access,
            imports=full_access,
            categories=full_access,
            brands=full_access
        )
    
    # ADMIN - Accès large sauf gestion des utilisateurs
    elif role == UserRole.ADMIN:
        admin_access = ModulePermission(view=True, create=True, edit=True, delete=True, approve=True)
        view_only = ModulePermission(view=True)
        return UserPermissions(
            dashboard=admin_access,
            products=admin_access,
            stock=admin_access,
            orders=admin_access,
            clients=admin_access,
            workshop=admin_access,
            finance=admin_access,
            fiscal=admin_access,
            hr=admin_access,
            ged=admin_access,
            reports=admin_access,
            users=view_only,  # Peut voir mais pas créer/modifier des users
            imports=admin_access,
            categories=admin_access,
            brands=admin_access
        )
    
    # MANAGER - Accès modéré (peut approuver mais pas supprimer)
    elif role == UserRole.MANAGER:
        manager_access = ModulePermission(view=True, create=True, edit=True, delete=False, approve=True)
        return UserPermissions(
            dashboard=ModulePermission(view=True),
            products=manager_access,
            stock=manager_access,
            orders=manager_access,
            clients=manager_access,
            workshop=manager_access,
            finance=ModulePermission(view=True),
            fiscal=ModulePermission(view=True),
            hr=manager_access,
            ged=manager_access,
            reports=ModulePermission(view=True),
            users=ModulePermission(view=True),
            imports=manager_access,
            categories=ModulePermission(view=True, create=True, edit=True),
            brands=ModulePermission(view=True, create=True, edit=True)
        )
    
    # CLIENT - Accès très limité (portail client uniquement)
    else:  # UserRole.CLIENT
        return UserPermissions(
            dashboard=ModulePermission(view=True),
            products=ModulePermission(view=True),
            stock=ModulePermission(),
            orders=ModulePermission(view=True, create=True),
            clients=ModulePermission(),
            workshop=ModulePermission(),
            finance=ModulePermission(),
            fiscal=ModulePermission(),
            hr=ModulePermission(),
            ged=ModulePermission(),
            reports=ModulePermission(),
            users=ModulePermission(),
            imports=ModulePermission(),
            categories=ModulePermission(view=True),
            brands=ModulePermission(view=True)
        )


def get_user_permissions(user: dict) -> UserPermissions:
    """
    Récupère les permissions effectives d'un utilisateur
    Si des permissions personnalisées sont définies, les utilise
    Sinon, utilise les permissions par défaut du rôle
    """
    # Si l'utilisateur a des permissions personnalisées, les utiliser
    if user.get("permissions"):
        return UserPermissions(**user["permissions"])
    
    # Sinon, utiliser les permissions par défaut du rôle
    role = UserRole(user.get("role", "client"))
    return get_default_permissions(role)


def check_permission(user: dict, module: str, action: str) -> bool:
    """
    Vérifie si un utilisateur a la permission pour une action sur un module
    
    Args:
        user: Dict contenant les infos de l'utilisateur
        module: Nom du module (ex: "hr", "products", "finance")
        action: Action à vérifier (ex: "view", "create", "edit", "delete", "approve")
    
    Returns:
        bool: True si l'utilisateur a la permission, False sinon
    """
    # SUPERADMIN a toujours accès
    if user.get("role") == UserRole.SUPERADMIN.value:
        return True
    
    # Récupérer les permissions de l'utilisateur
    permissions = get_user_permissions(user)
    
    # Vérifier si le module existe
    module_permissions = getattr(permissions, module, None)
    if not module_permissions:
        return False
    
    # Vérifier si l'action est autorisée
    return getattr(module_permissions, action, False)


def has_any_permission(user: dict, module: str) -> bool:
    """
    Vérifie si l'utilisateur a au moins une permission sur le module
    """
    return check_permission(user, module, "view")


# Templates de permissions prédéfinis pour faciliter l'assignation
PERMISSION_TEMPLATES = {
    "hr_manager": {
        "name": "Responsable RH",
        "description": "Accès complet au module RH avec approbation",
        "permissions": UserPermissions(
            dashboard=ModulePermission(view=True),
            hr=ModulePermission(view=True, create=True, edit=True, delete=True, approve=True),
            reports=ModulePermission(view=True)
        )
    },
    "hr_agent": {
        "name": "Agent RH",
        "description": "Accès RH limité sans approbation",
        "permissions": UserPermissions(
            dashboard=ModulePermission(view=True),
            hr=ModulePermission(view=True, create=True, edit=True, delete=False, approve=False),
            reports=ModulePermission(view=True)
        )
    },
    "accountant": {
        "name": "Comptable",
        "description": "Accès Finance et Fiscalité",
        "permissions": UserPermissions(
            dashboard=ModulePermission(view=True),
            finance=ModulePermission(view=True, create=True, edit=True, delete=False, approve=True),
            fiscal=ModulePermission(view=True, create=True, edit=True, delete=False, approve=True),
            reports=ModulePermission(view=True)
        )
    },
    "workshop_manager": {
        "name": "Chef d'Atelier",
        "description": "Gestion complète de l'atelier",
        "permissions": UserPermissions(
            dashboard=ModulePermission(view=True),
            workshop=ModulePermission(view=True, create=True, edit=True, delete=True, approve=True),
            products=ModulePermission(view=True),
            stock=ModulePermission(view=True, create=True, edit=True),
            clients=ModulePermission(view=True),
            reports=ModulePermission(view=True)
        )
    },
    "stock_manager": {
        "name": "Gestionnaire Stock",
        "description": "Gestion du stock et produits",
        "permissions": UserPermissions(
            dashboard=ModulePermission(view=True),
            products=ModulePermission(view=True, create=True, edit=True, delete=True),
            stock=ModulePermission(view=True, create=True, edit=True, delete=True),
            imports=ModulePermission(view=True, create=True, edit=True),
            categories=ModulePermission(view=True, create=True, edit=True),
            brands=ModulePermission(view=True, create=True, edit=True),
            reports=ModulePermission(view=True)
        )
    },
    "sales_manager": {
        "name": "Responsable Commercial",
        "description": "Gestion des ventes et clients",
        "permissions": UserPermissions(
            dashboard=ModulePermission(view=True),
            orders=ModulePermission(view=True, create=True, edit=True, delete=False, approve=True),
            clients=ModulePermission(view=True, create=True, edit=True, delete=False),
            products=ModulePermission(view=True),
            reports=ModulePermission(view=True)
        )
    }
}
